(ns clojurewerkz.eventoverse.collector.web.live-feed
  (:require [noir.core :refer [defpage]]
            [clojure.tools.logging :as log]
            [clojurewerkz.eventoverse.collector.core :refer [add-consumer]]
            [clojure.data.json :as json]
            [clojurewerkz.eventoverse.collector.web.common :refer [in-layout
                                                                   format-tag format-tags tags-from
                                                                   format-emitted-at format-hostname
                                                                   inject-custom-attributes]]
            [stencil.core :as st])
  (:use noir-async.core))


(defrecord FeedWatcher
    [conn ^String app ^String env])

(defn format-event
  [m ^String app ^String env]
  (-> m
      (format-tags app env)
      (format-emitted-at)
      (format-hostname app env)
      (inject-custom-attributes)))

(defn render-row
  [m ^String app ^String env]
  (st/render-file "templates/events/event_row" (format-event m app env)))


(def feed-consumers
  (atom #{}))

(defn peer-connected
  [conn ^String app ^String env]
  (swap! feed-consumers conj (FeedWatcher. conn app env))
  (log/infof "New live feed connection (total %d)" (count @feed-consumers)))

(defn peer-disconnected
  [conn ^String app ^String env]
  (swap! feed-consumers disj (FeedWatcher. conn app env))
  (log/infof "Live feed connection closed (total %d)" (count @feed-consumers)))

(defn propagate
  [msg ^String app ^String env]
  (doseq [watcher (filter (fn [feed-watcher]
                            (and (= (.app feed-watcher) app)
                                 (= (.env feed-watcher) env))) @feed-consumers)]
    (async-push (:conn watcher) (json/json-str {:type "events.new"
                                                :data msg
                                                :html (render-row msg app env)}))))

(add-consumer propagate)

(defpage-async [:get "/apps/:app/env/:env/live/feed/"]
  {:keys [^String app ^String env]} conn
  (peer-connected conn app env)
  (on-close conn (fn []
                   (peer-disconnected conn app env))))
