(ns clojurewerkz.eventoverse.collector.server
  (:gen-class)
  (:require [clojure.tools.cli :refer [cli]]
            [orcette.conf          :as oc]
            [clojure.tools.logging :as log]
            [langohr.core          :as lh]
            [monger.core           :as mg]
            [monger.collection     :as mc]
            [noir.server           :as web]
            [slingshot.slingshot :refer [try+]]
            [clojurewerkz.quartzite.scheduler         :as quartz.scheduler]
            [clojurewerkz.elastisch.rest              :as es]
            [clojurewerkz.elastisch.rest.index        :as es.idx]
            [clojurewerkz.eventoverse.collector.amqp  :as amqp]
            [clojurewerkz.eventoverse.collector.store :as store])
  (:use aleph.http
        lamina.core)
  (:import [com.mongodb MongoOptions ServerAddress]))


;;
;; Implementation
;;

(def config {})

(defn- exit!
  "Logs an error and exits with non-zero code"
  [s]
  (log/error s)
  (System/exit 1))

(defn- ensure-indexes-on
  [^String collection]
  (log/infof "Ensuring %s has indexes..." collection)
  (mc/ensure-index collection {:type 1})
  (mc/ensure-index collection {:emitted_at 1})
  (mc/ensure-index collection {:tags 1})
  (mc/ensure-index collection {:hostname 1}))

(defn- initialize-mongodb-collections
  [options]
  (doseq [{:keys [name environments]} (:applications options)]
    (doseq [e environments]
      (let [collection (store/events-bucket-name-for name e)]
        (ensure-indexes-on collection)))))

(defn initialize-mongodb
  [options]
  (when-let [mongodb-opts (:mongodb options)]
    (let [host                (get-in mongodb-opts [:connection :host])
          port                (get-in mongodb-opts [:connection :port])
          ^ServerAddress sa   (mg/server-address host port)
          ^MongoOptions  opts (mg/mongo-options :threads-allowed-to-block-for-connection-multiplier 2048 :auto-connect-retry true)
          db                  (:database mongodb-opts)]
      (mg/connect! sa opts)
      (mg/use-db! db)
      (log/infof "Connected to MongoDB at %s, database name is %s" db host)
      (initialize-mongodb-collections options))))

(defn initialize-amqp
  [options]
  (when-let [amqp-opts (:amqp options)]
    (let [vhost     (get-in amqp-opts [:connection :vhost])
          username  (get-in amqp-opts [:connection :username])]
      (log/infof "Connecting to RabbitMQ as %s@%s" username vhost)
      (amqp/connect (:connection amqp-opts))
      (log/info "Connected to RabbitMQ"))))


(defn- initialize-elasticsearch-index
  [^String s]
  (log/infof "Initializing search index %s..." s)
  ;; ElasticSearch responds with 400s if an index already exists
  (when-not (es.idx/exists? s)
    (es.idx/create s :mappings {:event {:properties {:type       {:type "string" :store "yes" :index "not_analyzed"}
                                                     :emitted_at {:type "date" :store "yes"}
                                                     ;; arrays get automatically detected
                                                     :tags       {:type "string" :store "yes"}
                                                     :hostname   {:type "string" :store "yes" :index "not_analyzed"}}}})))

(defn- initialize-elasticsearch-indexes
  [options]
  (doseq [{:keys [name environments]} (:applications options)]
    (doseq [e environments]
      (let [index (store/events-bucket-name-for name e)]
        (initialize-elasticsearch-index index)))))

(defn initialize-elasticsearch
  [options]
  (if-let [endpoint (-> options :elasticsearch :url)]
    (do
      (es/connect! endpoint)
      (log/infof "Initialized ElasticSearch")
      (initialize-elasticsearch-indexes options))
    (log/warnf "ElasticSearch connection is not configured!")))

(defn- noir-mode-from
  [options]
  (let [env (keyword (get options :environment :development))]
    (case env
      :dev         :dev
      :development :dev
      :test        :test
      :staging     :production
      :production  :production)))

(defn initialize-custom
  [options]
  (if-let [initializer-config (get-in options [:custom-intializer])]
    (do
      (require (eval (:ns initializer-config)))
      (let [funk (ns-resolve (eval (:ns initializer-config)) (eval (:fn initializer-config)))]
        (funk  options)))))

(defn initialize-webserver
  [options]
  (.start (Thread. ^Runnable (fn []
                               (log/info "Loading Noir views")
                               (web/load-views-ns 'clojurewerkz.eventoverse.collector.web)
                               (when (get-in options [:custom-view-nss])
                                 (doseq [n (get-in options [:custom-view-nss])]
                                   (web/load-views-ns (eval n))))
                               (let [mode   (noir-mode-from options)
                                     port   (-> options :http_server :port)
                                     noir-f (web/gen-handler {:mode mode
                                                              :ns 'clojurewerkz.eventoverse.collector})]
                                 (log/infof "Starting Web server on port %d in %s mode" port mode)
                                 (start-http-server
                                  (wrap-ring-handler noir-f) {:port port :websocket true}))))))

(def initializers [initialize-mongodb
                   initialize-elasticsearch
                   initialize-webserver
                   initialize-amqp])



;;
;; API
;;

(defn start
  [options]
  (alter-var-root (var config) (constantly options))
  (dorun (map #(% options) initializers)))

(defn -main
  [& args]
  (let [[options positional-args banner] (cli args
                                              ["--environment" "Environment to run in (development, production, et cetera)" :default "development"]
                                              ["--config-file" "Path to configuration file to use"])]
    (if-let [config (oc/config-for options)]
      (do
        (log/infof "Using config file at %s" (oc/config-path-for options))
        (start config))
      (exit! "Please either specify environment using --environment switch or config file path using --config-file switch"))
    nil))
