(ns clojurewerkz.eventoverse.collector.amqp
  (:require [clojure.tools.logging :as log]
            [clojure.stacktrace    :as strace]
            [langohr.core      :as lhc]
            [langohr.queue     :as lhq]
            [langohr.exchange  :as lhe]
            [langohr.basic     :as lhb]
            [clojure.data.json :as json]
            [langohr.consumers :as lhcons]
            [clojurewerkz.eventoverse.collector.core :as proc])
  (:import [com.rabbitmq.client Connection Channel AMQP$BasicProperties QueueingConsumer$Delivery]))

;;
;; Implementation
;;

(def ^Connection  amqp-connection)
(def ^Channel     collector-channel)

(def ^:const events-exchange-name "events")
(def ^:const collector-queue-name "events.collector")


(defn- open-channels
  [connection]
  (alter-var-root (var collector-channel) (constantly
                                           (lhc/create-channel connection))))

(defn- declare-entities
  [ch]
  (lhe/declare ch events-exchange-name "fanout" :durable true)
  (lhq/declare ch collector-queue-name :durable true :auto-delete false :exclusive false)
  (lhq/bind    ch collector-queue-name events-exchange-name))

(defn- start-consumer
  [ch queue f]
  (log/infof "Starting a blocking consumer on queue %s" queue)
  (lhcons/subscribe ch queue f :auto-ack false))


;;
;; API
;;

(defn event-handler
  "Handles messages that represent events"
  [ch {:keys [type delivery-tag] :as meta} ^bytes payload]
  (log/infof "Handling a message of type %s" type)
  (try
    (proc/process (merge (json/read-json (String. payload) true)
                         {:type type}))
    (catch Exception e
      (log/errorf "Handling a message of type %s resulted in an exception: %s" type e)
      (.printStackTrace e))
    (finally
      (lhb/ack collector-channel delivery-tag))))

(defn connect
  [options]
  (alter-var-root (var amqp-connection) (constantly (lhc/connect options)))
  (open-channels amqp-connection)
  (declare-entities collector-channel)
  (start-consumer collector-channel collector-queue-name event-handler)
  amqp-connection)

(defn purge-events-queue!
  []
  (lhq/purge collector-channel collector-queue-name))
