(ns clojurewerkz.eventoverse.client.udp
  (:require [cheshire.core :as json]
            clojurewerkz.eventoverse.client
            clojurewerkz.eventoverse.json)
  (:use [clj-time.core :only [now]])
  (:import org.joda.time.DateTime
           [java.net DatagramPacket DatagramSocket InetAddress]))

(def ^{:dynamic true}
  *default-client*)

(defn publish-event*
  [client event-type attributes]
  (let [msg (json/generate-string (merge (.defaults client)
                                         attributes
                                         {:type event-type
                                          :emitted_at ^DateTime (now)}))
        buf (.getBytes msg)]
    (.send
     (.socket client)
     (DatagramPacket. buf
                      (count buf)
                      (InetAddress/getByName (.host client))
                      (.port client)))))

(deftype UdpClient [^String host ^int port ^DatagramSocket socket defaults])

(extend-protocol clojurewerkz.eventoverse.client/Client
  UdpClient
  (publish-event [client event-type attributes]
    (publish-event* client event-type attributes))
  (trace  [client msg]
    (publish-event* client "log.trace" {:message msg}))
  (debug  [client msg]
    (publish-event* client "log.debug" {:message msg}))
  (info   [client msg]
    (publish-event* client "log.info"  {:message msg}))
  (warn   [client msg]
    (publish-event* client "log.warn"  {:message msg}))
  (error  [client msg]
    (publish-event* client "log.error" {:message msg}))
  (fatal  [client msg]
    (publish-event* client "log.fatal" {:message msg}))
  (tracef [client msg & rest]
    (publish-event* client "log.trace" {:message (apply format (cons msg rest))}))
  (debugf [client msg & rest]
    (publish-event* client "log.debug" {:message (apply format (cons msg rest))}))
  (infof  [client msg & rest]
    (publish-event* client "log.info"  {:message (apply format (cons msg rest))}))
  (warnf  [client msg & rest]
    (publish-event* client "log.warn"  {:message (apply format (cons msg rest))}))
  (errorf [client msg & rest]
    (publish-event* client "log.error" {:message (apply format (cons msg rest))}))
  (fatalf [client msg & rest]
    (publish-event* client "log.fatal" {:message (apply format (cons msg rest))})))

(defn make
  [defaults options]
  (UdpClient. (:host options) (:port options) (DatagramSocket.) defaults))

(defn initialize! [defaults options]
  (let [client (make defaults options)]
    (alter-var-root (var *default-client*) (constantly client))
    client))
