(ns clojurewerkz.eventoverse.client.amqp
  (:require [langohr.core      :as lhc]
            [langohr.basic     :as lhb]
            [clojure.data.json :as json]
            [clojurewerkz.eventoverse.client])
  (:use monger.joda-time
        [clj-time.core :only [now]])
  (:import [com.rabbitmq.client Connection Channel]
           java.util.Date))

;;
;; Implementation
;;

(def ^{:const true} exchange-name "events")
;; events exchange is a fanout so routing key is not used.
(def ^{:const true} routing-key "")

(defn publish-event*
  [client event-type attributes]
  (lhb/publish (.channel client) exchange-name routing-key (json/json-str (merge (.defaults client)
                                                                                             attributes
                                                                                             {:emitted_at (now)}))
               :type event-type))

;;
;; API
;;

(def ^{:dynamic true}
  *default-client*)


(deftype AMQPClient [^Connection connection ^Channel channel defaults]
  clojurewerkz.eventoverse.client/Client
  (publish-event [client event-type attributes]
    (publish-event* client event-type attributes))
  (trace  [client msg]
    (publish-event* client "log.trace" {:message msg}))
  (debug  [client msg]
    (publish-event* client "log.debug" {:message msg}))
  (info   [client msg]
    (publish-event* client "log.info"  {:message msg}))
  (warn   [client msg]
    (publish-event* client "log.warn"  {:message msg}))
  (error  [client msg]
    (publish-event* client "log.error" {:message msg}))
  (fatal  [client msg]
    (publish-event* client "log.fatal" {:message msg}))
  (tracef [client msg & rest]
    (publish-event* client "log.trace" {:message (apply format (cons msg rest))}))
  (debugf [client msg & rest]
    (publish-event* client "log.debug" {:message (apply format (cons msg rest))}))
  (infof  [client msg & rest]
    (publish-event* client "log.info"  {:message (apply format (cons msg rest))}))
  (warnf  [client msg & rest]
    (publish-event* client "log.warn"  {:message (apply format (cons msg rest))}))
  (errorf [client msg & rest]
    (publish-event* client "log.error" {:message (apply format (cons msg rest))}))
  (fatalf [client msg & rest]
    (publish-event* client "log.fatal" {:message (apply format (cons msg rest))})))


(defn make [defaults options]
  (let [connection (lhc/connect options)
        channel    (lhc/create-channel connection)]
    (AMQPClient. connection channel defaults)))

(defn initialize! [defaults options]
  (let [client (make defaults options)]
    (alter-var-root (var *default-client* client))
    client))
