(ns clojurewerkz.eventoverse.client.in-memory
  (:require [clojure.data.json :as json]
            [clojurewerkz.eventoverse.client])
  (:use [clj-time.core :only [now]])
  (:import java.util.Date))

(defn publish-event*
  [client event-type attributes]
  (swap! (.store client) conj (merge (.defaults client)
                                    attributes
                                    {:emitted_at (now) :type event-type})))

;;
;; API
;;

(def ^{:dynamic true}
  *default-client*)

(deftype InMemoryClient [defaults store]
  clojurewerkz.eventoverse.client/Client
  (publish-event [client event-type attributes]
    (publish-event* client event-type attributes))
  (trace  [client msg]
    (publish-event* client "log.trace" {:message msg}))
  (debug  [client msg]
    (publish-event* client "log.debug" {:message msg}))
  (info   [client msg]
    (publish-event* client "log.info"  {:message msg}))
  (warn   [client msg]
    (publish-event* client "log.warn"  {:message msg}))
  (error  [client msg]
    (publish-event* client "log.error" {:message msg}))
  (fatal  [client msg]
    (publish-event* client "log.fatal" {:message msg}))
  (tracef [client msg & rest]
    (publish-event* client "log.trace" {:message (apply format (cons msg rest))}))
  (debugf [client msg & rest]
    (publish-event* client "log.debug" {:message (apply format (cons msg rest))}))
  (infof  [client msg & rest]
    (publish-event* client "log.info"  {:message (apply format (cons msg rest))}))
  (warnf  [client msg & rest]
    (publish-event* client "log.warn"  {:message (apply format (cons msg rest))}))
  (errorf [client msg & rest]
    (publish-event* client "log.error" {:message (apply format (cons msg rest))}))
  (fatalf [client msg & rest]
    (publish-event* client "log.fatal" {:message (apply format (cons msg rest))}))

  java.lang.Iterable
  (iterator [client]
    (.iterator @(.store client)))

  clojure.lang.Counted
  (count [client]
    (clojure.core/count @(.store client)))

  clojure.lang.IDeref
  (deref [client]
    @(.store client))


  Object
  (toString [_] (str @store)))

(defn make [defaults]
  (InMemoryClient. defaults (atom [])))

(defn initialize! [defaults]
  (let [client (make defaults)]
    (alter-var-root (var *default-client* client))
    client))
