(ns clojurewerkz.elastisch.rest.document
  (:refer-clojure :exclude [get replace count sort])
  (:require [clojurewerkz.elastisch.rest :as rest]
            [cheshire.core :as json]
            [clojure.string :as string]
            [clojure.set :refer :all]
            [clojurewerkz.elastisch.rest.utils :refer [join-names]]
            [clojurewerkz.elastisch.rest.response :refer [not-found? hits-from]]))

;;
;; API
;;

(defn create
  "Adds document to the search index. Document id will be generated automatically.

   Options:

     * :id (string): unique document id. If not provided, it will be generated by ElasticSearch
     * :timestamp (string): document timestamp either as millis since the epoch,
                                          or, in the configured date format
     * :ttl (long): document TTL in milliseconds. Must be > 0
     * :refresh (boolean, default: false): should a refresh be executed post this index operation?
     * :version (long): document version
     * :version-type (string, default: \"internal\"): \"internal\" or \"external\"
     * :content-type (string): document content type
     * :routing (string): controls the shard routing of the request. Using this value to hash the shard
                          and not the id
     * :percolate (string): the percolate query to use to reduce the percolated queries that are going to run against this doc.
                           Can be set to \"*\" which means \"all queries\"
     * :parent (string): parent document id

   Examples:

   (require '[clojurewerkz.elastisch.rest.document :as doc])

   (doc/create \"people\" \"person\" {:first-name \"John\" :last-name \"Appleseed\" :age 28})

   (doc/create \"people\" \"person\" {:first-name \"John\" :last-name \"Appleseed\" :age 28} :id \"1825c5432775b8d1a477acfae57e91ac8c767aed\")"
  ([index mapping-type document & {:as params}]
     (rest/post (rest/mapping-type-url index mapping-type) :body document :query-params params)))

(defn put
  "Creates or updates a document in the search index, using the provided document id"
  ([index mapping-type id document]
     (rest/put (rest/record-url index mapping-type id) :body document))
  ([index mapping-type id document & {:as params}]
     (rest/put (rest/record-url index mapping-type id) :body document :query-params params)))

(defn update-with-script
  "Updates a document using a script"
  ([index mapping-type id script]
     (rest/post (rest/record-update-url index mapping-type id) :body {:script script}))
  ([index mapping-type id script params]
     (rest/post (rest/record-update-url index mapping-type id)
                :body {:script script :params params})))

(defn get
  "Fetches and returns a document by id or nil if it does not exist.

   Examples:

   (require '[clojurewerkz.elastisch.rest.document :as doc])

   (doc/get \"people\" \"person\" \"1825c5432775b8d1a477acfae57e91ac8c767aed\")"
  [index mapping-type id & {:as params}]
  (let [result (rest/get (rest/record-url index mapping-type id) :query-params params)]
    (if (not-found? result)
      nil
      result)))

(defn delete
  "Deletes document from the index.

   Related ElasticSearch documentation guide: http://www.elasticsearch.org/guide/reference/api/delete.html"
  ([index mapping-type id]
     (rest/delete (rest/record-url index mapping-type id)))
  ([index mapping-type id & {:as params}]
     (rest/delete (rest/record-url index mapping-type id) :query-params params)))

(defn present?
  "Returns true if a document with the given id is present in the provided index
   with the given mapping type."
  [index mapping-type id]
  (not (nil? (get index mapping-type id))))

(defn multi-get
  "Multi get returns only documents that are found (exist).

   Queries can passed as a collection of maps with three keys: :_index,
   :_type and :_id:

   (doc/multi-get [{:_index index-name :_type mapping-type :_id \"1\"}
                   {:_index index-name :_type mapping-type :_id \"2\"}])


   2-argument version accepts an index name that eliminates the need to include
   :_index in every query map:

   (doc/multi-get index-name [{:_type mapping-type :_id \"1\"}
                              {:_type mapping-type :_id \"2\"}])

   3-argument version also accepts a mapping type that eliminates the need to include
   :_type in every query map:

   (doc/multi-get index-name mapping-type [{:_id \"1\"}
                                           {:_id \"2\"}])"
  ([query]
     (let [results (rest/post (rest/index-mget-url)
                              :body {:docs query})]
       (filter :exists (:docs results))))
  ([index query]
     (let [results (rest/post (rest/index-mget-url index)
                              :body {:docs query})]
       (filter :exists (:docs results))))
  ([index mapping-type query]
     (let [results (rest/post (rest/index-mget-url index mapping-type)
                              :body {:docs query})]
       (filter :exists (:docs results)))))

(defn search
  "Performs a search query across one or more indexes and one or more mapping types.

   Passing index name as \"_all\" means searching across all indexes.

   Examples:

   (require '[clojurewerkz.elastisch.rest.document :as doc])
   (require '[clojurewerkz.elastisch.query :as q])

   (doc/search \"people\" \"person\" :query (q/prefix :username \"appl\"))"
  [index mapping-type & {:as options}]
  (let [qk   [:search_type :scroll :routing :preference :ignore_indices]
        qp   (select-keys options qk)
        body (apply dissoc (concat [options] qk))]
    (rest/post (rest/search-url (join-names index)
                                (join-names mapping-type))
               :body body
               :query-params qp)))

(defn search-all-types
  "Performs a search query across one or more indexes and all mapping types."
  [index & {:as options}]
  (let [qk   [:search_type :scroll :routing :preference :ignore_indices]
        qp   (select-keys options qk)
        body (apply dissoc (concat [options] qk))]
    (rest/post (rest/search-url (join-names index))
               :body body
               :query-params qp)))

(defn search-all-indexes-and-types
  "Performs a search query across all indexes and all mapping types. This may put very high load on your
   ElasticSearch cluster so use this function with care."
  [& {:as options}]
  (let [qk   [:search_type :scroll :routing :preference]
        qp   (select-keys options qk)
        body (apply dissoc (concat [options] qk))]
    (rest/post (rest/search-url)
               :body body
               :query-params qp)))

(defn scroll
  "Performs a scroll query, fetching the next page of results from a
   query given a scroll id"
  [scroll-id & {:as options}]
  (let [qk   [:search_type :scroll :routing :preference]
        qp   (assoc (select-keys options qk) :scroll_id scroll-id)
        body (apply dissoc (concat [options] qk))]
    (rest/get (rest/scroll-url)
               :query-params qp)))

(defn scroll-seq
  "Returns a lazy sequence of all documents for a given scroll query"
  [prev-resp]
  (let [hits      (hits-from prev-resp)
        scroll-id (:_scroll_id prev-resp)]
    (if (seq hits)
      (concat hits (lazy-seq (scroll-seq (scroll scroll-id :scroll "1m"))))
      hits)))

(defn replace
  "Replaces document with given id with a new one"
  [index mapping-type id document]
  (delete index mapping-type id)
  (put index mapping-type id document))


(defn count
  "Performs a count query.

   Examples:

   (require '[clojurewerkz.elastisch.rest.document :as doc])
   (require '[clojurewerkz.elastisch.query :as q])

   (doc/count \"people\" \"person\")
   (doc/count \"people\" \"person\" (q/prefix :username \"appl\"))

   Related ElasticSearch documentation guide: http://www.elasticsearch.org/guide/reference/api/count.html"
  ([index mapping-type]
     (rest/get (rest/count-url (join-names index) (join-names mapping-type))))
  ([index mapping-type query]
     (rest/post (rest/count-url (join-names index) (join-names mapping-type)) :body query))
  ([index mapping-type query & { :as options }]
     (rest/post (rest/count-url (join-names index) (join-names mapping-type))
                :query-params (select-keys options [:df :analyzer :default_operator :ignore_indices])
                :body query)))

(def ^{:doc "Optional parameters that all query-based delete functions share"
       :const true}
  optional-delete-query-parameters [:df :analyzer :default_operator :consistency])

(defn delete-by-query
  "Performs a delete-by-query operation.

   Related ElasticSearch documentation guide: http://www.elasticsearch.org/guide/reference/api/delete-by-query.html"
  ([index mapping-type query]
     (rest/delete (rest/delete-by-query-url (join-names index) (join-names mapping-type)) :body query))
  ([index mapping-type query & { :as options }]
     (rest/delete (rest/delete-by-query-url (join-names index) (join-names mapping-type))
                  :query-params (select-keys options (conj optional-delete-query-parameters :ignore_indices))
                  :body query)))

(defn delete-by-query-across-all-types
  "Performs a delete-by-query operation across all mapping types.

   Related ElasticSearch documentation guide: http://www.elasticsearch.org/guide/reference/api/delete-by-query.html"
  ([index query]
     (rest/delete (rest/delete-by-query-url (join-names index)) :body query))
  ([index query & {:as options}]
     (rest/delete (rest/delete-by-query-url (join-names index))
                  :query-params (select-keys options (conj optional-delete-query-parameters :ignore_indices))
                  :body query)))

(defn delete-by-query-across-all-indexes-and-types
  "Performs a delete-by-query operation across all indexes and mapping types.
   This may put very high load on your ElasticSearch cluster so use this function with care.

   Related ElasticSearch documentation guide: http://www.elasticsearch.org/guide/reference/api/delete-by-query.html"
  ([query]
     (rest/delete (rest/delete-by-query-url) :body query))
  ([query & {:as options}]
     (rest/delete (rest/delete-by-query-url)
                  :query-params (select-keys options optional-delete-query-parameters)
                  :body query)))


(defn more-like-this
  "Performs a More Like This (MLT) query.

   Related ElasticSearch documentation guide: http://www.elasticsearch.org/guide/reference/api/more-like-this.html"
  [index mapping-type id &{:as params}]
  (rest/get (rest/more-like-this-url index mapping-type id)
            :query-params params))

(defn validate-query
  "Validates a query without actually executing it. Has the same API as clojurewerkz.elastisch.rest.document/search
   but does not take the mapping type parameter."
  [index query & {:as options}]
  (rest/get (rest/query-validation-url index) :body (json/encode query) :query-params options))


(defn analyze 
  "see http://www.elasticsearch.org/guide/en/elasticsearch/reference/current/indices-analyze.html

   Examples:

   (require '[clojurewerkz.elastisch.rest.document :as doc])

   (doc/analyze \"foo bar baz\")
   (doc/analyze \"foo bar baz\" :index \"some-index-name\")
   (doc/analyze \"foo bar baz\" :analyzer \"whitespace\")
   (doc/analyze \"foo bar baz\" :index \"some-index-name\" :field \"some-field-name\")"
  ([text & {:as params}] (rest/get (rest/analyze-url (:index params))
                                   :query-params (assoc params :text text))))




