(ns jdk.sql.ResultSet
  "A table of data representing a database result set, which
  is usually generated by executing a statement that queries the database.

  A ResultSet object  maintains a cursor pointing
  to its current row of data.  Initially the cursor is positioned
  before the first row. The next method moves the
  cursor to the next row, and because it returns false
  when there are no more rows in the ResultSet object,
  it can be used in a while loop to iterate through
  the result set.

  A default ResultSet object is not updatable and
  has a cursor that moves forward only.  Thus, you can
  iterate through it only once and only from the first row to the
  last row. It is possible to
  produce ResultSet objects that are scrollable and/or
  updatable.  The following code fragment, in which con
  is a valid Connection object, illustrates how to make
  a result set that is scrollable and insensitive to updates by others, and
  that is updatable. See ResultSet fields for other
  options.


        Statement stmt = con.createStatement(
                                       ResultSet.TYPE_SCROLL_INSENSITIVE,
                                       ResultSet.CONCUR_UPDATABLE);
        ResultSet rs = stmt.executeQuery(`SELECT a, b FROM TABLE2`);
        // rs will be scrollable, will not show changes made by others,
        // and will be updatable
  The ResultSet interface provides
  getter methods (getBoolean, getLong, and so on)
  for retrieving column values from the current row.
  Values can be retrieved using either the index number of the
  column or the name of the column.  In general, using the
  column index will be more efficient.  Columns are numbered from 1.
  For maximum portability, result set columns within each row should be
  read in left-to-right order, and each column should be read only once.

  For the getter methods, a JDBC driver attempts
  to convert the underlying data to the Java type specified in the
  getter method and returns a suitable Java value.  The JDBC specification
  has a table showing the allowable mappings from SQL types to Java types
  that can be used by the ResultSet getter methods.

  Column names used as input to getter methods are case
  insensitive.  When a getter method is called  with
  a column name and several columns have the same name,
  the value of the first matching column will be returned.
  The column name option is
  designed to be used when column names are used in the SQL
  query that generated the result set.
  For columns that are NOT explicitly named in the query, it
  is best to use column numbers. If column names are used, the
  programmer should take care to guarantee that they uniquely refer to
  the intended columns, which can be assured with the SQL AS clause.

  A set of updater methods were added to this interface
  in the JDBC 2.0 API (Java™ 2 SDK,
  Standard Edition, version 1.2). The comments regarding parameters
  to the getter methods also apply to parameters to the
  updater methods.

  The updater methods may be used in two ways:

  to update a column value in the current row.  In a scrollable
      ResultSet object, the cursor can be moved backwards
      and forwards, to an absolute position, or to a position
      relative to the current row.
      The following code fragment updates the NAME column
      in the fifth row of the ResultSet object
      rs and then uses the method updateRow
      to update the data source table from which rs was derived.


        rs.absolute(5); // moves the cursor to the fifth row of rs
        rs.updateString(`NAME`, `AINSWORTH`); // updates the
           // NAME column of row 5 to be AINSWORTH
        rs.updateRow(); // updates the row in the data source
  to insert column values into the insert row.  An updatable
      ResultSet object has a special row associated with
      it that serves as a staging area for building a row to be inserted.
      The following code fragment moves the cursor to the insert row, builds
      a three-column row, and inserts it into rs and into
      the data source table using the method insertRow.


        rs.moveToInsertRow(); // moves cursor to the insert row
        rs.updateString(1, `AINSWORTH`); // updates the
           // first column of the insert row to be AINSWORTH
        rs.updateInt(2,35); // updates the second column to be 35
        rs.updateBoolean(3, true); // updates the third column to true
        rs.insertRow();
        rs.moveToCurrentRow();

  A ResultSet object is automatically closed when the
  Statement object that
  generated it is closed, re-executed, or used
  to retrieve the next result from a sequence of multiple results.

  The number, types and properties of a ResultSet
  object's columns are provided by the ResultSetMetaData
  object returned by the ResultSet.getMetaData method."
  (:refer-clojure :only [require comment defn ->])
  (:import [java.sql ResultSet]))

(defn row-updated
  "Retrieves whether the current row has been updated.  The value returned
   depends on whether or not the result set can detect updates.

   Note: Support for the rowUpdated method is optional with a result set
   concurrency of CONCUR_READ_ONLY

  returns: true if the current row is detected to
   have been visibly updated by the owner or another; false otherwise - `boolean`

  throws: java.sql.SQLException - if a database access error occurs or this method is called on a closed result set"
  ([^java.sql.ResultSet this]
    (-> this (.rowUpdated))))

(defn get-binary-stream
  "Retrieves the value of the designated column in the current row
   of this ResultSet object as a  stream of
   uninterpreted bytes. The value can then be read in chunks from the
   stream. This method is particularly
   suitable for retrieving large LONGVARBINARY values.

   Note: All the data in the returned stream must be
   read prior to getting the value of any other column. The next
   call to a getter method implicitly closes the stream.  Also, a
   stream may return 0 when the method
   InputStream.available
   is called whether there is data available or not.

  column-index - the first column is 1, the second is 2, ... - `int`

  returns: a Java input stream that delivers the database column value
           as a stream of uninterpreted bytes;
           if the value is SQL NULL, the value returned is
           null - `java.io.InputStream`

  throws: java.sql.SQLException - if the columnIndex is not valid; if a database access error occurs or this method is called on a closed result set"
  ([^java.sql.ResultSet this ^Integer column-index]
    (-> this (.getBinaryStream column-index))))

(defn set-fetch-size
  "Gives the JDBC driver a hint as to the number of rows that should
   be fetched from the database when more rows are needed for this
   ResultSet object.
   If the fetch size specified is zero, the JDBC driver
   ignores the value and is free to make its own best guess as to what
   the fetch size should be.  The default value is set by the
   Statement object
   that created the result set.  The fetch size may be changed at any time.

  rows - the number of rows to fetch - `int`

  throws: java.sql.SQLException - if a database access error occurs; this method is called on a closed result set or the condition rows >= 0 is not satisfied"
  ([^java.sql.ResultSet this ^Integer rows]
    (-> this (.setFetchSize rows))))

(defn get-n-character-stream
  "Retrieves the value of the designated column in the current row
   of this ResultSet object as a
   java.io.Reader object.
   It is intended for use when
   accessing  NCHAR,NVARCHAR
   and LONGNVARCHAR columns.

  column-index - the first column is 1, the second is 2, ... - `int`

  returns: a java.io.Reader object that contains the column
   value; if the value is SQL NULL, the value returned is
   null in the Java programming language. - `java.io.Reader`

  throws: java.sql.SQLException - if the columnIndex is not valid; if a database access error occurs or this method is called on a closed result set"
  ([^java.sql.ResultSet this ^Integer column-index]
    (-> this (.getNCharacterStream column-index))))

(defn get-sqlxml
  "Retrieves the value of the designated column in  the current row of
    this ResultSet as a
   java.sql.SQLXML object in the Java programming language.

  column-index - the first column is 1, the second is 2, ... - `int`

  returns: a SQLXML object that maps an SQL XML value - `java.sql.SQLXML`

  throws: java.sql.SQLException - if the columnIndex is not valid; if a database access error occurs or this method is called on a closed result set"
  ([^java.sql.ResultSet this ^Integer column-index]
    (-> this (.getSQLXML column-index))))

(defn get-unicode-stream
  "Deprecated. use getCharacterStream in place of
                getUnicodeStream

  column-index - the first column is 1, the second is 2, ... - `int`

  returns: a Java input stream that delivers the database column value
           as a stream of two-byte Unicode characters;
           if the value is SQL NULL, the value returned is
           null - `java.lang..io.InputStream`

  throws: java.sql.SQLException - if the columnIndex is not valid; if a database access error occurs or this method is called on a closed result set"
  ([^java.sql.ResultSet this ^Integer column-index]
    (-> this (.getUnicodeStream column-index))))

(defn get-row-id
  "Retrieves the value of the designated column in the current row of this
   ResultSet object as a java.sql.RowId object in the Java
   programming language.

  column-index - the first column is 1, the second 2, ... - `int`

  returns: the column value; if the value is a SQL NULL the
       value returned is null - `java.sql.RowId`

  throws: java.sql.SQLException - if the columnIndex is not valid; if a database access error occurs or this method is called on a closed result set"
  ([^java.sql.ResultSet this ^Integer column-index]
    (-> this (.getRowId column-index))))

(defn get-ascii-stream
  "Retrieves the value of the designated column in the current row
   of this ResultSet object as
   a stream of ASCII characters. The value can then be read in chunks from the
   stream. This method is particularly
   suitable for retrieving large LONGVARCHAR values.
   The JDBC driver will
   do any necessary conversion from the database format into ASCII.

   Note: All the data in the returned stream must be
   read prior to getting the value of any other column. The next
   call to a getter method implicitly closes the stream.  Also, a
   stream may return 0 when the method
   InputStream.available
   is called whether there is data available or not.

  column-index - the first column is 1, the second is 2, ... - `int`

  returns: a Java input stream that delivers the database column value
   as a stream of one-byte ASCII characters;
   if the value is SQL NULL, the
   value returned is null - `java.io.InputStream`

  throws: java.sql.SQLException - if the columnIndex is not valid; if a database access error occurs or this method is called on a closed result set"
  ([^java.sql.ResultSet this ^Integer column-index]
    (-> this (.getAsciiStream column-index))))

(defn update-date
  "Updates the designated column with a java.sql.Date value.
   The updater methods are used to update column values in the
   current row or the insert row.  The updater methods do not
   update the underlying database; instead the updateRow or
   insertRow methods are called to update the database.

  column-index - the first column is 1, the second is 2, ... - `int`
  x - the new column value - `java.sql.Date`

  throws: java.sql.SQLException - if the columnIndex is not valid; if a database access error occurs; the result set concurrency is CONCUR_READ_ONLY or this method is called on a closed result set"
  ([^java.sql.ResultSet this ^Integer column-index ^java.sql.Date x]
    (-> this (.updateDate column-index x))))

(defn get-byte
  "Retrieves the value of the designated column in the current row
   of this ResultSet object as
   a byte in the Java programming language.

  column-index - the first column is 1, the second is 2, ... - `int`

  returns: the column value; if the value is SQL NULL, the
   value returned is 0 - `byte`

  throws: java.sql.SQLException - if the columnIndex is not valid; if a database access error occurs or this method is called on a closed result set"
  ([^java.sql.ResultSet this ^Integer column-index]
    (-> this (.getByte column-index))))

(defn update-float
  "Updates the designated column with a float value.
   The updater methods are used to update column values in the
   current row or the insert row.  The updater methods do not
   update the underlying database; instead the updateRow or
   insertRow methods are called to update the database.

  column-index - the first column is 1, the second is 2, ... - `int`
  x - the new column value - `float`

  throws: java.sql.SQLException - if the columnIndex is not valid; if a database access error occurs; the result set concurrency is CONCUR_READ_ONLY or this method is called on a closed result set"
  ([^java.sql.ResultSet this ^Integer column-index ^Float x]
    (-> this (.updateFloat column-index x))))

(defn next
  "Moves the cursor forward one row from its current position.
   A ResultSet cursor is initially positioned
   before the first row; the first call to the method
   next makes the first row the current row; the
   second call makes the second row the current row, and so on.

   When a call to the next method returns false,
   the cursor is positioned after the last row. Any
   invocation of a ResultSet method which requires a
   current row will result in a SQLException being thrown.
    If the result set type is TYPE_FORWARD_ONLY, it is vendor specified
   whether their JDBC driver implementation will return false or
    throw an SQLException on a
   subsequent call to next.

   If an input stream is open for the current row, a call
   to the method next will
   implicitly close it. A ResultSet object's
   warning chain is cleared when a new row is read.

  returns: true if the new current row is valid;
   false if there are no more rows - `boolean`

  throws: java.sql.SQLException - if a database access error occurs or this method is called on a closed result set"
  ([^java.sql.ResultSet this]
    (-> this (.next))))

(defn update-timestamp
  "Updates the designated column with a java.sql.Timestamp
   value.
   The updater methods are used to update column values in the
   current row or the insert row.  The updater methods do not
   update the underlying database; instead the updateRow or
   insertRow methods are called to update the database.

  column-index - the first column is 1, the second is 2, ... - `int`
  x - the new column value - `java.sql.Timestamp`

  throws: java.sql.SQLException - if the columnIndex is not valid; if a database access error occurs; the result set concurrency is CONCUR_READ_ONLY or this method is called on a closed result set"
  ([^java.sql.ResultSet this ^Integer column-index ^java.sql.Timestamp x]
    (-> this (.updateTimestamp column-index x))))

(defn get-ref
  "Retrieves the value of the designated column in the current row
   of this ResultSet object as a Ref object
   in the Java programming language.

  column-index - the first column is 1, the second is 2, ... - `int`

  returns: a Ref object representing an SQL REF
           value - `java.sql.Ref`

  throws: java.sql.SQLException - if the columnIndex is not valid; if a database access error occurs or this method is called on a closed result set"
  ([^java.sql.ResultSet this ^Integer column-index]
    (-> this (.getRef column-index))))

(defn before-first?
  "Retrieves whether the cursor is before the first row in
   this ResultSet object.

   Note:Support for the isBeforeFirst method
   is optional for ResultSets with a result
   set type of TYPE_FORWARD_ONLY

  returns: true if the cursor is before the first row;
   false if the cursor is at any other position or the
   result set contains no rows - `boolean`

  throws: java.sql.SQLException - if a database access error occurs or this method is called on a closed result set"
  ([^java.sql.ResultSet this]
    (-> this (.isBeforeFirst))))

(defn get-type
  "Retrieves the type of this ResultSet object.
   The type is determined by the Statement object
   that created the result set.

  returns: ResultSet.TYPE_FORWARD_ONLY,
           ResultSet.TYPE_SCROLL_INSENSITIVE,
           or ResultSet.TYPE_SCROLL_SENSITIVE - `int`

  throws: java.sql.SQLException - if a database access error occurs or this method is called on a closed result set"
  ([^java.sql.ResultSet this]
    (-> this (.getType))))

(defn update-array
  "Updates the designated column with a java.sql.Array value.
   The updater methods are used to update column values in the
   current row or the insert row.  The updater methods do not
   update the underlying database; instead the updateRow or
   insertRow methods are called to update the database.

  column-index - the first column is 1, the second is 2, ... - `int`
  x - the new column value - `java.sql.Array`

  throws: java.sql.SQLException - if the columnIndex is not valid; if a database access error occurs; the result set concurrency is CONCUR_READ_ONLY or this method is called on a closed result set"
  ([^java.sql.ResultSet this ^Integer column-index ^java.sql.Array x]
    (-> this (.updateArray column-index x))))

(defn move-to-current-row
  "Moves the cursor to the remembered cursor position, usually the
   current row.  This method has no effect if the cursor is not on
   the insert row.

  throws: java.sql.SQLException - if a database access error occurs; this method is called on a closed result set or the result set concurrency is CONCUR_READ_ONLY"
  ([^java.sql.ResultSet this]
    (-> this (.moveToCurrentRow))))

(defn update-ref
  "Updates the designated column with a java.sql.Ref value.
   The updater methods are used to update column values in the
   current row or the insert row.  The updater methods do not
   update the underlying database; instead the updateRow or
   insertRow methods are called to update the database.

  column-index - the first column is 1, the second is 2, ... - `int`
  x - the new column value - `java.sql.Ref`

  throws: java.sql.SQLException - if the columnIndex is not valid; if a database access error occurs; the result set concurrency is CONCUR_READ_ONLY or this method is called on a closed result set"
  ([^java.sql.ResultSet this ^Integer column-index ^java.sql.Ref x]
    (-> this (.updateRef column-index x))))

(defn get-n-string
  "Retrieves the value of the designated column in the current row
   of this ResultSet object as
   a String in the Java programming language.
   It is intended for use when
   accessing  NCHAR,NVARCHAR
   and LONGNVARCHAR columns.

  column-index - the first column is 1, the second is 2, ... - `int`

  returns: the column value; if the value is SQL NULL, the
   value returned is null - `java.lang.String`

  throws: java.sql.SQLException - if the columnIndex is not valid; if a database access error occurs or this method is called on a closed result set"
  ([^java.sql.ResultSet this ^Integer column-index]
    (-> this (.getNString column-index))))

(defn update-big-decimal
  "Updates the designated column with a java.math.BigDecimal
   value.
   The updater methods are used to update column values in the
   current row or the insert row.  The updater methods do not
   update the underlying database; instead the updateRow or
   insertRow methods are called to update the database.

  column-index - the first column is 1, the second is 2, ... - `int`
  x - the new column value - `java.math.BigDecimal`

  throws: java.sql.SQLException - if the columnIndex is not valid; if a database access error occurs; the result set concurrency is CONCUR_READ_ONLY or this method is called on a closed result set"
  ([^java.sql.ResultSet this ^Integer column-index ^java.math.BigDecimal x]
    (-> this (.updateBigDecimal column-index x))))

(defn get-string
  "Retrieves the value of the designated column in the current row
   of this ResultSet object as
   a String in the Java programming language.

  column-index - the first column is 1, the second is 2, ... - `int`

  returns: the column value; if the value is SQL NULL, the
   value returned is null - `java.lang.String`

  throws: java.sql.SQLException - if the columnIndex is not valid; if a database access error occurs or this method is called on a closed result set"
  ([^java.sql.ResultSet this ^Integer column-index]
    (-> this (.getString column-index))))

(defn get-short
  "Retrieves the value of the designated column in the current row
   of this ResultSet object as
   a short in the Java programming language.

  column-index - the first column is 1, the second is 2, ... - `int`

  returns: the column value; if the value is SQL NULL, the
   value returned is 0 - `short`

  throws: java.sql.SQLException - if the columnIndex is not valid; if a database access error occurs or this method is called on a closed result set"
  ([^java.sql.ResultSet this ^Integer column-index]
    (-> this (.getShort column-index))))

(defn update-row
  "Updates the underlying database with the new contents of the
   current row of this ResultSet object.
   This method cannot be called when the cursor is on the insert row.

  throws: java.sql.SQLException - if a database access error occurs; the result set concurrency is CONCUR_READ_ONLY; this method is called on a closed result set or if this method is called when the cursor is on the insert row"
  ([^java.sql.ResultSet this]
    (-> this (.updateRow))))

(defn update-null
  "Updates the designated column with a null value.

   The updater methods are used to update column values in the
   current row or the insert row.  The updater methods do not
   update the underlying database; instead the updateRow
   or insertRow methods are called to update the database.

  column-index - the first column is 1, the second is 2, ... - `int`

  throws: java.sql.SQLException - if the columnIndex is not valid; if a database access error occurs; the result set concurrency is CONCUR_READ_ONLY or this method is called on a closed result set"
  ([^java.sql.ResultSet this ^Integer column-index]
    (-> this (.updateNull column-index))))

(defn clear-warnings
  "Clears all warnings reported on this ResultSet object.
   After this method is called, the method getWarnings
   returns null until a new warning is
   reported for this ResultSet object.

  throws: java.sql.SQLException - if a database access error occurs or this method is called on a closed result set"
  ([^java.sql.ResultSet this]
    (-> this (.clearWarnings))))

(defn get-timestamp
  "Retrieves the value of the designated column in the current row
   of this ResultSet object as a java.sql.Timestamp object
   in the Java programming language.
   This method uses the given calendar to construct an appropriate millisecond
   value for the timestamp if the underlying database does not store
   timezone information.

  column-index - the first column is 1, the second is 2, ... - `int`
  cal - the java.util.Calendar object to use in constructing the timestamp - `java.util.Calendar`

  returns: the column value as a java.sql.Timestamp object;
   if the value is SQL NULL,
   the value returned is null in the Java programming language - `java.sql.Timestamp`

  throws: java.sql.SQLException - if the columnIndex is not valid; if a database access error occurs or this method is called on a closed result set"
  ([^java.sql.ResultSet this ^Integer column-index ^java.util.Calendar cal]
    (-> this (.getTimestamp column-index cal)))
  ([^java.sql.ResultSet this ^Integer column-index]
    (-> this (.getTimestamp column-index))))

(defn get-warnings
  "Retrieves the first warning reported by calls on this
   ResultSet object.
   Subsequent warnings on this ResultSet object
   will be chained to the SQLWarning object that
   this method returns.

   The warning chain is automatically cleared each time a new
   row is read.  This method may not be called on a ResultSet
   object that has been closed; doing so will cause an
   SQLException to be thrown.

   Note: This warning chain only covers warnings caused
   by ResultSet methods.  Any warning caused by
   Statement methods
   (such as reading OUT parameters) will be chained on the
   Statement object.

  returns: the first SQLWarning object reported or
           null if there are none - `java.sql.SQLWarning`

  throws: java.sql.SQLException - if a database access error occurs or this method is called on a closed result set"
  ([^java.sql.ResultSet this]
    (-> this (.getWarnings))))

(defn update-blob
  "Updates the designated column using the given input stream, which
   will have the specified number of bytes.


   The updater methods are used to update column values in the
   current row or the insert row.  The updater methods do not
   update the underlying database; instead the updateRow or
   insertRow methods are called to update the database.

  column-index - the first column is 1, the second is 2, ... - `int`
  input-stream - An object that contains the data to set the parameter value to. - `java.io.InputStream`
  length - the number of bytes in the parameter data. - `long`

  throws: java.sql.SQLException - if the columnIndex is not valid; if a database access error occurs; the result set concurrency is CONCUR_READ_ONLY or this method is called on a closed result set"
  ([^java.sql.ResultSet this ^Integer column-index ^java.io.InputStream input-stream ^Long length]
    (-> this (.updateBlob column-index input-stream length)))
  ([^java.sql.ResultSet this ^Integer column-index ^java.sql.Blob x]
    (-> this (.updateBlob column-index x))))

(defn update-n-clob
  "Updates the designated column using the given Reader
   object, which is the given number of characters long.
   When a very large UNICODE value is input to a LONGVARCHAR
   parameter, it may be more practical to send it via a
   java.io.Reader object. The JDBC driver will
   do any necessary conversion from UNICODE to the database char format.


   The updater methods are used to update column values in the
   current row or the insert row.  The updater methods do not
   update the underlying database; instead the updateRow or
   insertRow methods are called to update the database.

  column-index - the first column is 1, the second 2, ... - `int`
  reader - An object that contains the data to set the parameter value to. - `java.io.Reader`
  length - the number of characters in the parameter data. - `long`

  throws: java.sql.SQLException - if the columnIndex is not valid; if the driver does not support national character sets; if the driver can detect that a data conversion error could occur; this method is called on a closed result set, if a database access error occurs or the result set concurrency is CONCUR_READ_ONLY"
  ([^java.sql.ResultSet this ^Integer column-index ^java.io.Reader reader ^Long length]
    (-> this (.updateNClob column-index reader length)))
  ([^java.sql.ResultSet this ^Integer column-index ^java.sql.NClob n-clob]
    (-> this (.updateNClob column-index n-clob))))

(defn get-object
  "Retrieves the value of the designated column in the current row
   of this ResultSet object as an Object
   in the Java programming language.
   If the value is an SQL NULL,
   the driver returns a Java null.
   This method uses the given Map object
   for the custom mapping of the
   SQL structured or distinct type that is being retrieved.

  column-index - the first column is 1, the second is 2, ... - `int`
  map - a java.util.Map object that contains the mapping from SQL type names to classes in the Java programming language - `java.util.Map<java.lang.String,java.lang.Class<?>>`

  returns: an Object in the Java programming language
   representing the SQL value - `java.lang.Object`

  throws: java.sql.SQLException - if the columnIndex is not valid; if a database access error occurs or this method is called on a closed result set"
  ([^java.sql.ResultSet this ^Integer column-index ^java.util.Map> map]
    (-> this (.getObject column-index map)))
  ([^java.sql.ResultSet this ^Integer column-index]
    (-> this (.getObject column-index))))

(defn get-fetch-direction
  "Retrieves the fetch direction for this
   ResultSet object.

  returns: the current fetch direction for this ResultSet object - `int`

  throws: java.sql.SQLException - if a database access error occurs or this method is called on a closed result set"
  ([^java.sql.ResultSet this]
    (-> this (.getFetchDirection))))

(defn get-bytes
  "Retrieves the value of the designated column in the current row
   of this ResultSet object as
   a byte array in the Java programming language.
   The bytes represent the raw values returned by the driver.

  column-index - the first column is 1, the second is 2, ... - `int`

  returns: the column value; if the value is SQL NULL, the
   value returned is null - `byte[]`

  throws: java.sql.SQLException - if the columnIndex is not valid; if a database access error occurs or this method is called on a closed result set"
  ([^java.sql.ResultSet this ^Integer column-index]
    (-> this (.getBytes column-index))))

(defn get-statement
  "Retrieves the Statement object that produced this
   ResultSet object.
   If the result set was generated some other way, such as by a
   DatabaseMetaData method, this method  may return
   null.

  returns: the Statement object that produced
   this ResultSet object or null
   if the result set was produced some other way - `java.sql.Statement`

  throws: java.sql.SQLException - if a database access error occurs or this method is called on a closed result set"
  ([^java.sql.ResultSet this]
    (-> this (.getStatement))))

(defn get-clob
  "Retrieves the value of the designated column in the current row
   of this ResultSet object as a Clob object
   in the Java programming language.

  column-index - the first column is 1, the second is 2, ... - `int`

  returns: a Clob object representing the SQL
           CLOB value in the specified column - `java.sql.Clob`

  throws: java.sql.SQLException - if the columnIndex is not valid; if a database access error occurs or this method is called on a closed result set"
  ([^java.sql.ResultSet this ^Integer column-index]
    (-> this (.getClob column-index))))

(defn get-blob
  "Retrieves the value of the designated column in the current row
   of this ResultSet object as a Blob object
   in the Java programming language.

  column-index - the first column is 1, the second is 2, ... - `int`

  returns: a Blob object representing the SQL
           BLOB value in the specified column - `java.sql.Blob`

  throws: java.sql.SQLException - if the columnIndex is not valid; if a database access error occurs or this method is called on a closed result set"
  ([^java.sql.ResultSet this ^Integer column-index]
    (-> this (.getBlob column-index))))

(defn get-date
  "Retrieves the value of the designated column in the current row
   of this ResultSet object as a java.sql.Date object
   in the Java programming language.
   This method uses the given calendar to construct an appropriate millisecond
   value for the date if the underlying database does not store
   timezone information.

  column-index - the first column is 1, the second is 2, ... - `int`
  cal - the java.util.Calendar object to use in constructing the date - `java.util.Calendar`

  returns: the column value as a java.sql.Date object;
   if the value is SQL NULL,
   the value returned is null in the Java programming language - `java.sql.Date`

  throws: java.sql.SQLException - if the columnIndex is not valid; if a database access error occurs or this method is called on a closed result set"
  ([^java.sql.ResultSet this ^Integer column-index ^java.util.Calendar cal]
    (-> this (.getDate column-index cal)))
  ([^java.sql.ResultSet this ^Integer column-index]
    (-> this (.getDate column-index))))

(defn update-string
  "Updates the designated column with a String value.
   The updater methods are used to update column values in the
   current row or the insert row.  The updater methods do not
   update the underlying database; instead the updateRow or
   insertRow methods are called to update the database.

  column-index - the first column is 1, the second is 2, ... - `int`
  x - the new column value - `java.lang.String`

  throws: java.sql.SQLException - if the columnIndex is not valid; if a database access error occurs; the result set concurrency is CONCUR_READ_ONLY or this method is called on a closed result set"
  ([^java.sql.ResultSet this ^Integer column-index ^java.lang.String x]
    (-> this (.updateString column-index x))))

(defn update-int
  "Updates the designated column with an int value.
   The updater methods are used to update column values in the
   current row or the insert row.  The updater methods do not
   update the underlying database; instead the updateRow or
   insertRow methods are called to update the database.

  column-index - the first column is 1, the second is 2, ... - `int`
  x - the new column value - `int`

  throws: java.sql.SQLException - if the columnIndex is not valid; if a database access error occurs; the result set concurrency is CONCUR_READ_ONLY or this method is called on a closed result set"
  ([^java.sql.ResultSet this ^Integer column-index ^Integer x]
    (-> this (.updateInt column-index x))))

(defn get-character-stream
  "Retrieves the value of the designated column in the current row
   of this ResultSet object as a
   java.io.Reader object.

  column-index - the first column is 1, the second is 2, ... - `int`

  returns: a java.io.Reader object that contains the column
   value; if the value is SQL NULL, the value returned is
   null in the Java programming language. - `java.io.Reader`

  throws: java.sql.SQLException - if the columnIndex is not valid; if a database access error occurs or this method is called on a closed result set"
  ([^java.sql.ResultSet this ^Integer column-index]
    (-> this (.getCharacterStream column-index))))

(defn get-long
  "Retrieves the value of the designated column in the current row
   of this ResultSet object as
   a long in the Java programming language.

  column-index - the first column is 1, the second is 2, ... - `int`

  returns: the column value; if the value is SQL NULL, the
   value returned is 0 - `long`

  throws: java.sql.SQLException - if the columnIndex is not valid; if a database access error occurs or this method is called on a closed result set"
  ([^java.sql.ResultSet this ^Integer column-index]
    (-> this (.getLong column-index))))

(defn update-boolean
  "Updates the designated column with a boolean value.
   The updater methods are used to update column values in the
   current row or the insert row.  The updater methods do not
   update the underlying database; instead the updateRow or
   insertRow methods are called to update the database.

  column-index - the first column is 1, the second is 2, ... - `int`
  x - the new column value - `boolean`

  throws: java.sql.SQLException - if the columnIndex is not valid; if a database access error occurs; the result set concurrency is CONCUR_READ_ONLY or this method is called on a closed result set"
  ([^java.sql.ResultSet this ^Integer column-index ^Boolean x]
    (-> this (.updateBoolean column-index x))))

(defn closed?
  "Retrieves whether this ResultSet object has been closed. A ResultSet is closed if the
   method close has been called on it, or if it is automatically closed.

  returns: true if this ResultSet object is closed; false if it is still open - `boolean`

  throws: java.sql.SQLException - if a database access error occurs"
  ([^java.sql.ResultSet this]
    (-> this (.isClosed))))

(defn get-cursor-name
  "Retrieves the name of the SQL cursor used by this ResultSet
   object.

   In SQL, a result table is retrieved through a cursor that is
   named. The current row of a result set can be updated or deleted
   using a positioned update/delete statement that references the
   cursor name. To insure that the cursor has the proper isolation
   level to support update, the cursor's SELECT statement
   should be of the form SELECT FOR UPDATE. If
   FOR UPDATE is omitted, the positioned updates may fail.

   The JDBC API supports this SQL feature by providing the name of the
   SQL cursor used by a ResultSet object.
   The current row of a ResultSet object
   is also the current row of this SQL cursor.

  returns: the SQL name for this ResultSet object's cursor - `java.lang.String`

  throws: java.sql.SQLException - if a database access error occurs or this method is called on a closed result set"
  ([^java.sql.ResultSet this]
    (-> this (.getCursorName))))

(defn update-ascii-stream
  "Updates the designated column with an ascii stream value, which will have
   the specified number of bytes.
   The updater methods are used to update column values in the
   current row or the insert row.  The updater methods do not
   update the underlying database; instead the updateRow or
   insertRow methods are called to update the database.

  column-index - the first column is 1, the second is 2, ... - `int`
  x - the new column value - `java.io.InputStream`
  length - the length of the stream - `int`

  throws: java.sql.SQLException - if the columnIndex is not valid; if a database access error occurs; the result set concurrency is CONCUR_READ_ONLY or this method is called on a closed result set"
  ([^java.sql.ResultSet this ^Integer column-index ^java.io.InputStream x ^Integer length]
    (-> this (.updateAsciiStream column-index x length)))
  ([^java.sql.ResultSet this ^Integer column-index ^java.io.InputStream x]
    (-> this (.updateAsciiStream column-index x))))

(defn get-int
  "Retrieves the value of the designated column in the current row
   of this ResultSet object as
   an int in the Java programming language.

  column-index - the first column is 1, the second is 2, ... - `int`

  returns: the column value; if the value is SQL NULL, the
   value returned is 0 - `int`

  throws: java.sql.SQLException - if the columnIndex is not valid; if a database access error occurs or this method is called on a closed result set"
  ([^java.sql.ResultSet this ^Integer column-index]
    (-> this (.getInt column-index))))

(defn previous
  "Moves the cursor to the previous row in this
   ResultSet object.

   When a call to the previous method returns false,
   the cursor is positioned before the first row.  Any invocation of a
   ResultSet method which requires a current row will result in a
   SQLException being thrown.

   If an input stream is open for the current row, a call to the method
   previous will implicitly close it.  A ResultSet
    object's warning change is cleared when a new row is read.

  returns: true if the cursor is now positioned on a valid row;
   false if the cursor is positioned before the first row - `boolean`

  throws: java.sql.SQLException - if a database access error occurs; this method is called on a closed result set or the result set type is TYPE_FORWARD_ONLY"
  ([^java.sql.ResultSet this]
    (-> this (.previous))))

(defn get-array
  "Retrieves the value of the designated column in the current row
   of this ResultSet object as an Array object
   in the Java programming language.

  column-index - the first column is 1, the second is 2, ... - `int`

  returns: an Array object representing the SQL
           ARRAY value in the specified column - `java.sql.Array`

  throws: java.sql.SQLException - if the columnIndex is not valid; if a database access error occurs or this method is called on a closed result set"
  ([^java.sql.ResultSet this ^Integer column-index]
    (-> this (.getArray column-index))))

(defn update-byte
  "Updates the designated column with a byte value.
   The updater methods are used to update column values in the
   current row or the insert row.  The updater methods do not
   update the underlying database; instead the updateRow or
   insertRow methods are called to update the database.

  column-index - the first column is 1, the second is 2, ... - `int`
  x - the new column value - `byte`

  throws: java.sql.SQLException - if the columnIndex is not valid; if a database access error occurs; the result set concurrency is CONCUR_READ_ONLY or this method is called on a closed result set"
  ([^java.sql.ResultSet this ^Integer column-index ^Byte x]
    (-> this (.updateByte column-index x))))

(defn update-double
  "Updates the designated column with a double value.
   The updater methods are used to update column values in the
   current row or the insert row.  The updater methods do not
   update the underlying database; instead the updateRow or
   insertRow methods are called to update the database.

  column-index - the first column is 1, the second is 2, ... - `int`
  x - the new column value - `double`

  throws: java.sql.SQLException - if the columnIndex is not valid; if a database access error occurs; the result set concurrency is CONCUR_READ_ONLY or this method is called on a closed result set"
  ([^java.sql.ResultSet this ^Integer column-index ^Double x]
    (-> this (.updateDouble column-index x))))

(defn update-long
  "Updates the designated column with a long value.
   The updater methods are used to update column values in the
   current row or the insert row.  The updater methods do not
   update the underlying database; instead the updateRow or
   insertRow methods are called to update the database.

  column-index - the first column is 1, the second is 2, ... - `int`
  x - the new column value - `long`

  throws: java.sql.SQLException - if the columnIndex is not valid; if a database access error occurs; the result set concurrency is CONCUR_READ_ONLY or this method is called on a closed result set"
  ([^java.sql.ResultSet this ^Integer column-index ^Long x]
    (-> this (.updateLong column-index x))))

(defn get-meta-data
  "Retrieves the  number, types and properties of
   this ResultSet object's columns.

  returns: the description of this ResultSet object's columns - `java.sql.ResultSetMetaData`

  throws: java.sql.SQLException - if a database access error occurs or this method is called on a closed result set"
  ([^java.sql.ResultSet this]
    (-> this (.getMetaData))))

(defn was-null
  "Reports whether
   the last column read had a value of SQL NULL.
   Note that you must first call one of the getter methods
   on a column to try to read its value and then call
   the method wasNull to see if the value read was
   SQL NULL.

  returns: true if the last column value read was SQL
           NULL and false otherwise - `boolean`

  throws: java.sql.SQLException - if a database access error occurs or this method is called on a closed result set"
  ([^java.sql.ResultSet this]
    (-> this (.wasNull))))

(defn update-time
  "Updates the designated column with a java.sql.Time value.
   The updater methods are used to update column values in the
   current row or the insert row.  The updater methods do not
   update the underlying database; instead the updateRow or
   insertRow methods are called to update the database.

  column-index - the first column is 1, the second is 2, ... - `int`
  x - the new column value - `java.sql.Time`

  throws: java.sql.SQLException - if the columnIndex is not valid; if a database access error occurs; the result set concurrency is CONCUR_READ_ONLY or this method is called on a closed result set"
  ([^java.sql.ResultSet this ^Integer column-index ^java.sql.Time x]
    (-> this (.updateTime column-index x))))

(defn first?
  "Retrieves whether the cursor is on the first row of
   this ResultSet object.

   Note:Support for the isFirst method
   is optional for ResultSets with a result
   set type of TYPE_FORWARD_ONLY

  returns: true if the cursor is on the first row;
   false otherwise - `boolean`

  throws: java.sql.SQLException - if a database access error occurs or this method is called on a closed result set"
  ([^java.sql.ResultSet this]
    (-> this (.isFirst))))

(defn get-row
  "Retrieves the current row number.  The first row is number 1, the
   second number 2, and so on.

   Note:Support for the getRow method
   is optional for ResultSets with a result
   set type of TYPE_FORWARD_ONLY

  returns: the current row number; 0 if there is no current row - `int`

  throws: java.sql.SQLException - if a database access error occurs or this method is called on a closed result set"
  ([^java.sql.ResultSet this]
    (-> this (.getRow))))

(defn find-column
  "Maps the given ResultSet column label to its
   ResultSet column index.

  column-label - the label for the column specified with the SQL AS clause. If the SQL AS clause was not specified, then the label is the name of the column - `java.lang.String`

  returns: the column index of the given column name - `int`

  throws: java.sql.SQLException - if the ResultSet object does not contain a column labeled columnLabel, a database access error occurs or this method is called on a closed result set"
  ([^java.sql.ResultSet this ^java.lang.String column-label]
    (-> this (.findColumn column-label))))

(defn close
  "Releases this ResultSet object's database and
   JDBC resources immediately instead of waiting for
   this to happen when it is automatically closed.

   The closing of a ResultSet object does not close the Blob,
   Clob or NClob objects created by the ResultSet. Blob,
   Clob or NClob objects remain valid for at least the duration of the
   transaction in which they are created, unless their free method is invoked.

   When a ResultSet is closed, any ResultSetMetaData
   instances that were created by calling the  getMetaData
   method remain accessible.

   Note: A ResultSet object
   is automatically closed by the
   Statement object that generated it when
   that Statement object is closed,
   re-executed, or is used to retrieve the next result from a
   sequence of multiple results.

   Calling the method close on a ResultSet
   object that is already closed is a no-op.

  throws: java.sql.SQLException - if a database access error occurs"
  ([^java.sql.ResultSet this]
    (-> this (.close))))

(defn before-first
  "Moves the cursor to the front of
   this ResultSet object, just before the
   first row. This method has no effect if the result set contains no rows.

  throws: java.sql.SQLException - if a database access error occurs; this method is called on a closed result set or the result set type is TYPE_FORWARD_ONLY"
  ([^java.sql.ResultSet this]
    (-> this (.beforeFirst))))

(defn last
  "Moves the cursor to the last row in
   this ResultSet object.

  returns: true if the cursor is on a valid row;
   false if there are no rows in the result set - `boolean`

  throws: java.sql.SQLException - if a database access error occurs; this method is called on a closed result set or the result set type is TYPE_FORWARD_ONLY"
  ([^java.sql.ResultSet this]
    (-> this (.last))))

(defn after-last
  "Moves the cursor to the end of
   this ResultSet object, just after the
   last row. This method has no effect if the result set contains no rows.

  throws: java.sql.SQLException - if a database access error occurs; this method is called on a closed result set or the result set type is TYPE_FORWARD_ONLY"
  ([^java.sql.ResultSet this]
    (-> this (.afterLast))))

(defn update-row-id
  "Updates the designated column with a RowId value. The updater
   methods are used to update column values in the current row or the insert
   row. The updater methods do not update the underlying database; instead
   the updateRow or insertRow methods are called
   to update the database.

  column-index - the first column is 1, the second 2, ... - `int`
  x - the column value - `java.sql.RowId`

  throws: java.sql.SQLException - if the columnIndex is not valid; if a database access error occurs; the result set concurrency is CONCUR_READ_ONLY or this method is called on a closed result set"
  ([^java.sql.ResultSet this ^Integer column-index ^java.sql.RowId x]
    (-> this (.updateRowId column-index x))))

(defn update-n-character-stream
  "Updates the designated column with a character stream value, which will have
   the specified number of bytes.   The
   driver does the necessary conversion from Java character format to
   the national character set in the database.
   It is intended for use when
   updating  NCHAR,NVARCHAR
   and LONGNVARCHAR columns.

   The updater methods are used to update column values in the
   current row or the insert row.  The updater methods do not
   update the underlying database; instead the updateRow or
   insertRow methods are called to update the database.

  column-index - the first column is 1, the second is 2, ... - `int`
  x - the new column value - `java.io.Reader`
  length - the length of the stream - `long`

  throws: java.sql.SQLException - if the columnIndex is not valid; if a database access error occurs; the result set concurrency is CONCUR_READ_ONLY or this method is called on a closed result set"
  ([^java.sql.ResultSet this ^Integer column-index ^java.io.Reader x ^Long length]
    (-> this (.updateNCharacterStream column-index x length)))
  ([^java.sql.ResultSet this ^Integer column-index ^java.io.Reader x]
    (-> this (.updateNCharacterStream column-index x))))

(defn refresh-row
  "Refreshes the current row with its most recent value in
   the database.  This method cannot be called when
   the cursor is on the insert row.

   The refreshRow method provides a way for an
   application to
   explicitly tell the JDBC driver to refetch a row(s) from the
   database.  An application may want to call refreshRow when
   caching or prefetching is being done by the JDBC driver to
   fetch the latest value of a row from the database.  The JDBC driver
   may actually refresh multiple rows at once if the fetch size is
   greater than one.

    All values are refetched subject to the transaction isolation
   level and cursor sensitivity.  If refreshRow is called after
   calling an updater method, but before calling
   the method updateRow, then the
   updates made to the row are lost.  Calling the method
   refreshRow frequently will likely slow performance.

  throws: java.sql.SQLException - if a database access error occurs; this method is called on a closed result set; the result set type is TYPE_FORWARD_ONLY or if this method is called when the cursor is on the insert row"
  ([^java.sql.ResultSet this]
    (-> this (.refreshRow))))

(defn move-to-insert-row
  "Moves the cursor to the insert row.  The current cursor position is
   remembered while the cursor is positioned on the insert row.

   The insert row is a special row associated with an updatable
   result set.  It is essentially a buffer where a new row may
   be constructed by calling the updater methods prior to
   inserting the row into the result set.

   Only the updater, getter,
   and insertRow methods may be
   called when the cursor is on the insert row.  All of the columns in
   a result set must be given a value each time this method is
   called before calling insertRow.
   An updater method must be called before a
   getter method can be called on a column value.

  throws: java.sql.SQLException - if a database access error occurs; this method is called on a closed result set or the result set concurrency is CONCUR_READ_ONLY"
  ([^java.sql.ResultSet this]
    (-> this (.moveToInsertRow))))

(defn update-n-string
  "Updates the designated column with a String value.
   It is intended for use when updating NCHAR,NVARCHAR
   and LONGNVARCHAR columns.
   The updater methods are used to update column values in the
   current row or the insert row.  The updater methods do not
   update the underlying database; instead the updateRow or
   insertRow methods are called to update the database.

  column-index - the first column is 1, the second 2, ... - `int`
  n-string - the value for the column to be updated - `java.lang.String`

  throws: java.sql.SQLException - if the columnIndex is not valid; if the driver does not support national character sets; if the driver can detect that a data conversion error could occur; this method is called on a closed result set; the result set concurrency is CONCUR_READ_ONLY or if a database access error occurs"
  ([^java.sql.ResultSet this ^Integer column-index ^java.lang.String n-string]
    (-> this (.updateNString column-index n-string))))

(defn update-object
  "Updates the designated column with an Object value.

   The updater methods are used to update column values in the
   current row or the insert row.  The updater methods do not
   update the underlying database; instead the updateRow or
   insertRow methods are called to update the database.

   If the second argument is an InputStream then the stream must contain
   the number of bytes specified by scaleOrLength.  If the second argument is a
   Reader then the reader must contain the number of characters specified
   by scaleOrLength. If these conditions are not true the driver will generate a
   SQLException when the statement is executed.

   The default implementation will throw SQLFeatureNotSupportedException

  column-index - the first column is 1, the second is 2, ... - `int`
  x - the new column value - `java.lang.Object`
  target-sql-type - the SQL type to be sent to the database - `java.sql.SQLType`
  scale-or-length - for an object of java.math.BigDecimal , this is the number of digits after the decimal point. For Java Object types InputStream and Reader, this is the length of the data in the stream or reader. For all other types, this value will be ignored. - `int`

  returns: `default void`

  throws: java.sql.SQLException - if the columnIndex is not valid; if a database access error occurs; the result set concurrency is CONCUR_READ_ONLY or this method is called on a closed result set"
  ([^java.sql.ResultSet this ^Integer column-index ^java.lang.Object x ^java.sql.SQLType target-sql-type ^Integer scale-or-length]
    (-> this (.updateObject column-index x target-sql-type scale-or-length)))
  ([^java.sql.ResultSet this ^Integer column-index ^java.lang.Object x ^Integer scale-or-length]
    (-> this (.updateObject column-index x scale-or-length)))
  ([^java.sql.ResultSet this ^Integer column-index ^java.lang.Object x]
    (-> this (.updateObject column-index x))))

(defn get-url
  "Retrieves the value of the designated column in the current row
   of this ResultSet object as a java.net.URL
   object in the Java programming language.

  column-index - the index of the column 1 is the first, 2 is the second,... - `int`

  returns: the column value as a java.net.URL object;
   if the value is SQL NULL,
   the value returned is null in the Java programming language - `java.net.URL`

  throws: java.sql.SQLException - if the columnIndex is not valid; if a database access error occurs; this method is called on a closed result set or if a URL is malformed"
  ([^java.sql.ResultSet this ^Integer column-index]
    (-> this (.getURL column-index))))

(defn get-fetch-size
  "Retrieves the fetch size for this
   ResultSet object.

  returns: the current fetch size for this ResultSet object - `int`

  throws: java.sql.SQLException - if a database access error occurs or this method is called on a closed result set"
  ([^java.sql.ResultSet this]
    (-> this (.getFetchSize))))

(defn get-concurrency
  "Retrieves the concurrency mode of this ResultSet object.
   The concurrency used is determined by the
   Statement object that created the result set.

  returns: the concurrency type, either
           ResultSet.CONCUR_READ_ONLY
           or ResultSet.CONCUR_UPDATABLE - `int`

  throws: java.sql.SQLException - if a database access error occurs or this method is called on a closed result set"
  ([^java.sql.ResultSet this]
    (-> this (.getConcurrency))))

(defn update-short
  "Updates the designated column with a short value.
   The updater methods are used to update column values in the
   current row or the insert row.  The updater methods do not
   update the underlying database; instead the updateRow or
   insertRow methods are called to update the database.

  column-index - the first column is 1, the second is 2, ... - `int`
  x - the new column value - `short`

  throws: java.sql.SQLException - if the columnIndex is not valid; if a database access error occurs; the result set concurrency is CONCUR_READ_ONLY or this method is called on a closed result set"
  ([^java.sql.ResultSet this ^Integer column-index ^Short x]
    (-> this (.updateShort column-index x))))

(defn get-boolean?
  "Retrieves the value of the designated column in the current row
   of this ResultSet object as
   a boolean in the Java programming language.

   If the designated column has a datatype of CHAR or VARCHAR
   and contains a `0` or has a datatype of BIT, TINYINT, SMALLINT, INTEGER or BIGINT
   and contains  a 0, a value of false is returned.  If the designated column has a datatype
   of CHAR or VARCHAR
   and contains a `1` or has a datatype of BIT, TINYINT, SMALLINT, INTEGER or BIGINT
   and contains  a 1, a value of true is returned.

  column-index - the first column is 1, the second is 2, ... - `int`

  returns: the column value; if the value is SQL NULL, the
   value returned is false - `boolean`

  throws: java.sql.SQLException - if the columnIndex is not valid; if a database access error occurs or this method is called on a closed result set"
  ([^java.sql.ResultSet this ^Integer column-index]
    (-> this (.getBoolean column-index))))

(defn get-n-clob
  "Retrieves the value of the designated column in the current row
   of this ResultSet object as a NClob object
   in the Java programming language.

  column-index - the first column is 1, the second is 2, ... - `int`

  returns: a NClob object representing the SQL
           NCLOB value in the specified column - `java.sql.NClob`

  throws: java.sql.SQLException - if the columnIndex is not valid; if the driver does not support national character sets; if the driver can detect that a data conversion error could occur; this method is called on a closed result set or if a database access error occurs"
  ([^java.sql.ResultSet this ^Integer column-index]
    (-> this (.getNClob column-index))))

(defn row-inserted
  "Retrieves whether the current row has had an insertion.
   The value returned depends on whether or not this
   ResultSet object can detect visible inserts.

   Note: Support for the rowInserted method is optional with a result set
   concurrency of CONCUR_READ_ONLY

  returns: true if the current row is detected to
   have been inserted; false otherwise - `boolean`

  throws: java.sql.SQLException - if a database access error occurs or this method is called on a closed result set"
  ([^java.sql.ResultSet this]
    (-> this (.rowInserted))))

(defn relative
  "Moves the cursor a relative number of rows, either positive or negative.
   Attempting to move beyond the first/last row in the
   result set positions the cursor before/after the
   the first/last row. Calling relative(0) is valid, but does
   not change the cursor position.

   Note: Calling the method relative(1)
   is identical to calling the method next() and
   calling the method relative(-1) is identical
   to calling the method previous().

  rows - an int specifying the number of rows to move from the current row; a positive number moves the cursor forward; a negative number moves the cursor backward - `int`

  returns: true if the cursor is on a row;
           false otherwise - `boolean`

  throws: java.sql.SQLException - if a database access error occurs; this method is called on a closed result set or the result set type is TYPE_FORWARD_ONLY"
  ([^java.sql.ResultSet this ^Integer rows]
    (-> this (.relative rows))))

(defn update-character-stream
  "Updates the designated column with a character stream value, which will have
   the specified number of bytes.
   The updater methods are used to update column values in the
   current row or the insert row.  The updater methods do not
   update the underlying database; instead the updateRow or
   insertRow methods are called to update the database.

  column-index - the first column is 1, the second is 2, ... - `int`
  x - the new column value - `java.io.Reader`
  length - the length of the stream - `int`

  throws: java.sql.SQLException - if the columnIndex is not valid; if a database access error occurs; the result set concurrency is CONCUR_READ_ONLY or this method is called on a closed result set"
  ([^java.sql.ResultSet this ^Integer column-index ^java.io.Reader x ^Integer length]
    (-> this (.updateCharacterStream column-index x length)))
  ([^java.sql.ResultSet this ^Integer column-index ^java.io.Reader x]
    (-> this (.updateCharacterStream column-index x))))

(defn after-last?
  "Retrieves whether the cursor is after the last row in
   this ResultSet object.

   Note:Support for the isAfterLast method
   is optional for ResultSets with a result
   set type of TYPE_FORWARD_ONLY

  returns: true if the cursor is after the last row;
   false if the cursor is at any other position or the
   result set contains no rows - `boolean`

  throws: java.sql.SQLException - if a database access error occurs or this method is called on a closed result set"
  ([^java.sql.ResultSet this]
    (-> this (.isAfterLast))))

(defn get-holdability
  "Retrieves the holdability of this ResultSet object

  returns: either ResultSet.HOLD_CURSORS_OVER_COMMIT or ResultSet.CLOSE_CURSORS_AT_COMMIT - `int`

  throws: java.sql.SQLException - if a database access error occurs or this method is called on a closed result set"
  ([^java.sql.ResultSet this]
    (-> this (.getHoldability))))

(defn update-clob
  "Updates the designated column using the given Reader
   object, which is the given number of characters long.
   When a very large UNICODE value is input to a LONGVARCHAR
   parameter, it may be more practical to send it via a
   java.io.Reader object. The JDBC driver will
   do any necessary conversion from UNICODE to the database char format.


   The updater methods are used to update column values in the
   current row or the insert row.  The updater methods do not
   update the underlying database; instead the updateRow or
   insertRow methods are called to update the database.

  column-index - the first column is 1, the second is 2, ... - `int`
  reader - An object that contains the data to set the parameter value to. - `java.io.Reader`
  length - the number of characters in the parameter data. - `long`

  throws: java.sql.SQLException - if the columnIndex is not valid; if a database access error occurs; the result set concurrency is CONCUR_READ_ONLY or this method is called on a closed result set"
  ([^java.sql.ResultSet this ^Integer column-index ^java.io.Reader reader ^Long length]
    (-> this (.updateClob column-index reader length)))
  ([^java.sql.ResultSet this ^Integer column-index ^java.sql.Clob x]
    (-> this (.updateClob column-index x))))

(defn update-binary-stream
  "Updates the designated column with a binary stream value, which will have
   the specified number of bytes.
   The updater methods are used to update column values in the
   current row or the insert row.  The updater methods do not
   update the underlying database; instead the updateRow or
   insertRow methods are called to update the database.

  column-index - the first column is 1, the second is 2, ... - `int`
  x - the new column value - `java.io.InputStream`
  length - the length of the stream - `int`

  throws: java.sql.SQLException - if the columnIndex is not valid; if a database access error occurs; the result set concurrency is CONCUR_READ_ONLY or this method is called on a closed result set"
  ([^java.sql.ResultSet this ^Integer column-index ^java.io.InputStream x ^Integer length]
    (-> this (.updateBinaryStream column-index x length)))
  ([^java.sql.ResultSet this ^Integer column-index ^java.io.InputStream x]
    (-> this (.updateBinaryStream column-index x))))

(defn update-bytes
  "Updates the designated column with a byte array value.
   The updater methods are used to update column values in the
   current row or the insert row.  The updater methods do not
   update the underlying database; instead the updateRow or
   insertRow methods are called to update the database.

  column-index - the first column is 1, the second is 2, ... - `int`
  x - the new column value - `byte[]`

  throws: java.sql.SQLException - if the columnIndex is not valid; if a database access error occurs; the result set concurrency is CONCUR_READ_ONLY or this method is called on a closed result set"
  ([^java.sql.ResultSet this ^Integer column-index x]
    (-> this (.updateBytes column-index x))))

(defn get-float
  "Retrieves the value of the designated column in the current row
   of this ResultSet object as
   a float in the Java programming language.

  column-index - the first column is 1, the second is 2, ... - `int`

  returns: the column value; if the value is SQL NULL, the
   value returned is 0 - `float`

  throws: java.sql.SQLException - if the columnIndex is not valid; if a database access error occurs or this method is called on a closed result set"
  ([^java.sql.ResultSet this ^Integer column-index]
    (-> this (.getFloat column-index))))

(defn get-time
  "Retrieves the value of the designated column in the current row
   of this ResultSet object as a java.sql.Time object
   in the Java programming language.
   This method uses the given calendar to construct an appropriate millisecond
   value for the time if the underlying database does not store
   timezone information.

  column-index - the first column is 1, the second is 2, ... - `int`
  cal - the java.util.Calendar object to use in constructing the time - `java.util.Calendar`

  returns: the column value as a java.sql.Time object;
   if the value is SQL NULL,
   the value returned is null in the Java programming language - `java.sql.Time`

  throws: java.sql.SQLException - if the columnIndex is not valid; if a database access error occurs or this method is called on a closed result set"
  ([^java.sql.ResultSet this ^Integer column-index ^java.util.Calendar cal]
    (-> this (.getTime column-index cal)))
  ([^java.sql.ResultSet this ^Integer column-index]
    (-> this (.getTime column-index))))

(defn get-double
  "Retrieves the value of the designated column in the current row
   of this ResultSet object as
   a double in the Java programming language.

  column-index - the first column is 1, the second is 2, ... - `int`

  returns: the column value; if the value is SQL NULL, the
   value returned is 0 - `double`

  throws: java.sql.SQLException - if the columnIndex is not valid; if a database access error occurs or this method is called on a closed result set"
  ([^java.sql.ResultSet this ^Integer column-index]
    (-> this (.getDouble column-index))))

(defn row-deleted
  "Retrieves whether a row has been deleted.  A deleted row may leave
   a visible `hole` in a result set.  This method can be used to
   detect holes in a result set.  The value returned depends on whether
   or not this ResultSet object can detect deletions.

   Note: Support for the rowDeleted method is optional with a result set
   concurrency of CONCUR_READ_ONLY

  returns: true if the current row is detected to
   have been deleted by the owner or another; false otherwise - `boolean`

  throws: java.sql.SQLException - if a database access error occurs or this method is called on a closed result set"
  ([^java.sql.ResultSet this]
    (-> this (.rowDeleted))))

(defn get-big-decimal
  "Deprecated. Use getBigDecimal(int columnIndex)
               or getBigDecimal(String columnLabel)

  column-index - the first column is 1, the second is 2, ... - `int`
  scale - the number of digits to the right of the decimal point - `int`

  returns: the column value; if the value is SQL NULL, the
   value returned is null - `java.lang..math.BigDecimal`

  throws: java.sql.SQLException - if the columnIndex is not valid; if a database access error occurs or this method is called on a closed result set"
  ([^java.sql.ResultSet this ^Integer column-index ^Integer scale]
    (-> this (.getBigDecimal column-index scale)))
  ([^java.sql.ResultSet this ^Integer column-index]
    (-> this (.getBigDecimal column-index))))

(defn set-fetch-direction
  "Gives a hint as to the direction in which the rows in this
   ResultSet object will be processed.
   The initial value is determined by the
   Statement object
   that produced this ResultSet object.
   The fetch direction may be changed at any time.

  direction - an int specifying the suggested fetch direction; one of ResultSet.FETCH_FORWARD, ResultSet.FETCH_REVERSE, or ResultSet.FETCH_UNKNOWN - `int`

  throws: java.sql.SQLException - if a database access error occurs; this method is called on a closed result set or the result set type is TYPE_FORWARD_ONLY and the fetch direction is not FETCH_FORWARD"
  ([^java.sql.ResultSet this ^Integer direction]
    (-> this (.setFetchDirection direction))))

(defn absolute
  "Moves the cursor to the given row number in
   this ResultSet object.

   If the row number is positive, the cursor moves to
   the given row number with respect to the
   beginning of the result set.  The first row is row 1, the second
   is row 2, and so on.

   If the given row number is negative, the cursor moves to
   an absolute row position with respect to
   the end of the result set.  For example, calling the method
   absolute(-1) positions the
   cursor on the last row; calling the method absolute(-2)
   moves the cursor to the next-to-last row, and so on.

   If the row number specified is zero, the cursor is moved to
   before the first row.

   An attempt to position the cursor beyond the first/last row in
   the result set leaves the cursor before the first row or after
   the last row.

   Note: Calling absolute(1) is the same
   as calling first(). Calling absolute(-1)
   is the same as calling last().

  row - the number of the row to which the cursor should move. A value of zero indicates that the cursor will be positioned before the first row; a positive number indicates the row number counting from the beginning of the result set; a negative number indicates the row number counting from the end of the result set - `int`

  returns: true if the cursor is moved to a position in this
   ResultSet object;
   false if the cursor is before the first row or after the
   last row - `boolean`

  throws: java.sql.SQLException - if a database access error occurs; this method is called on a closed result set or the result set type is TYPE_FORWARD_ONLY"
  ([^java.sql.ResultSet this ^Integer row]
    (-> this (.absolute row))))

(defn cancel-row-updates
  "Cancels the updates made to the current row in this
   ResultSet object.
   This method may be called after calling an
   updater method(s) and before calling
   the method updateRow to roll back
   the updates made to a row.  If no updates have been made or
   updateRow has already been called, this method has no
   effect.

  throws: java.sql.SQLException - if a database access error occurs; this method is called on a closed result set; the result set concurrency is CONCUR_READ_ONLY or if this method is called when the cursor is on the insert row"
  ([^java.sql.ResultSet this]
    (-> this (.cancelRowUpdates))))

(defn update-sqlxml
  "Updates the designated column with a java.sql.SQLXML value.
   The updater
   methods are used to update column values in the current row or the insert
   row. The updater methods do not update the underlying database; instead
   the updateRow or insertRow methods are called
   to update the database.

  column-index - the first column is 1, the second 2, ... - `int`
  xml-object - the value for the column to be updated - `java.sql.SQLXML`

  throws: java.sql.SQLException - if the columnIndex is not valid; if a database access error occurs; this method is called on a closed result set; the java.xml.transform.Result, Writer or OutputStream has not been closed for the SQLXML object; if there is an error processing the XML value or the result set concurrency is CONCUR_READ_ONLY. The getCause method of the exception may provide a more detailed exception, for example, if the stream does not contain valid XML."
  ([^java.sql.ResultSet this ^Integer column-index ^java.sql.SQLXML xml-object]
    (-> this (.updateSQLXML column-index xml-object))))

(defn delete-row
  "Deletes the current row from this ResultSet object
   and from the underlying database.  This method cannot be called when
   the cursor is on the insert row.

  throws: java.sql.SQLException - if a database access error occurs; the result set concurrency is CONCUR_READ_ONLY; this method is called on a closed result set or if this method is called when the cursor is on the insert row"
  ([^java.sql.ResultSet this]
    (-> this (.deleteRow))))

(defn first
  "Moves the cursor to the first row in
   this ResultSet object.

  returns: true if the cursor is on a valid row;
   false if there are no rows in the result set - `boolean`

  throws: java.sql.SQLException - if a database access error occurs; this method is called on a closed result set or the result set type is TYPE_FORWARD_ONLY"
  ([^java.sql.ResultSet this]
    (-> this (.first))))

(defn insert-row
  "Inserts the contents of the insert row into this
   ResultSet object and into the database.
   The cursor must be on the insert row when this method is called.

  throws: java.sql.SQLException - if a database access error occurs; the result set concurrency is CONCUR_READ_ONLY, this method is called on a closed result set, if this method is called when the cursor is not on the insert row, or if not all of non-nullable columns in the insert row have been given a non-null value"
  ([^java.sql.ResultSet this]
    (-> this (.insertRow))))

(defn last?
  "Retrieves whether the cursor is on the last row of
   this ResultSet object.
    Note: Calling the method isLast may be expensive
   because the JDBC driver
   might need to fetch ahead one row in order to determine
   whether the current row is the last row in the result set.

   Note: Support for the isLast method
   is optional for ResultSets with a result
   set type of TYPE_FORWARD_ONLY

  returns: true if the cursor is on the last row;
   false otherwise - `boolean`

  throws: java.sql.SQLException - if a database access error occurs or this method is called on a closed result set"
  ([^java.sql.ResultSet this]
    (-> this (.isLast))))

