(ns web.dom.History
  "The History interface allows manipulation of the browser session
  that is the pages visited in the tab or frame that the current
  is loaded in."
  (:refer-clojure :exclude [next]))

(defn back
  "Method.

  The back method moves back one page in the session history. If
  is no previous page, this method call does nothing.

  `window.history.back()`

  See also: `https://developer.mozilla.org/en-US/docs/Web/API/History/back`"
  [this ]
  (-> this (.back)))

(defn forward
  "Method.

  Moves forward one page in the session history. It has the same
  as calling history.go() with a delta parameter of 1.

  `window.history.forward();`

  See also: `https://developer.mozilla.org/en-US/docs/Web/API/History/forward`"
  [this ]
  (-> this (.forward)))

(defn go
  "Method.

  The go() method loads a specific page from the session history.
  can use it to move forwards and backwards through the history
  on the value of the delta parameter.

  `window.history.go(delta);`

  See also: `https://developer.mozilla.org/en-US/docs/Web/API/History/go`"
  [this delta]
  (-> this (.go delta)))

(defn push-state
  "Method.

  In an HTML document, the history.pushState() method adds a state
  the browsers history.

  `history.pushState(state, title, url);`

  See also: `https://developer.mozilla.org/en-US/docs/Web/API/History/pushState`"
  [this state title url]
  (-> this (.pushState state title url)))

(defn replace-state
  "Method.

  The replaceState() method modifies the current history entry,
  it with the state objects, title, and URL passed in the method
  This method is particularly useful when you want to update the
  object or URL of the current history entry in response to some
  action.

  `history.replaceState(stateObj, \\\"page 2\\\", \\\"bar.html\\\");`

  See also: `https://developer.mozilla.org/en-US/docs/Web/API/History/replaceState`"
  [this & args]
  (-> this .-replaceState (.apply this (clj->js args))))

(defn length
  "Property.

  [Read Only]

  The History.length read-only property returns an Integer representing
  number of elements in the session history, including the currently
  page. For example, for a page loaded in a new tab this property
  1.

  `length = history.length;`

  See also: `https://developer.mozilla.org/en-US/docs/Web/API/History/length`"
  [this]
  (-> this (.-length)))

(defn current
  "Property.

  Returns a DOMString representing the URL of the active item of
  session history. This property was never available to web content
  is no more supported by any browser. Use Location.href instead."
  [this]
  (-> this (.-current)))

(defn set-current!
  "Property.

  Returns a DOMString representing the URL of the active item of
  session history. This property was never available to web content
  is no more supported by any browser. Use Location.href instead."
  [this val]
  (aset this "current" val))

(defn next
  "Property.

  Returns a DOMString representing the URL of the next item in
  session history. This property was never available to web content
  is not supported by other browsers."
  [this]
  (-> this (.-next)))

(defn set-next!
  "Property.

  Returns a DOMString representing the URL of the next item in
  session history. This property was never available to web content
  is not supported by other browsers."
  [this val]
  (aset this "next" val))

(defn previous
  "Property.

  Returns a DOMString representing the URL of the previous item
  the session history. This property was never available to web
  and is not supported by other browsers."
  [this]
  (-> this (.-previous)))

(defn set-previous!
  "Property.

  Returns a DOMString representing the URL of the previous item
  the session history. This property was never available to web
  and is not supported by other browsers."
  [this val]
  (aset this "previous" val))

(defn scroll-restoration
  "Property.

  Allows web applications to explicitly set default scroll restoration
  on history navigation. This property can be either auto or manual."
  [this]
  (-> this (.-scrollRestoration)))

(defn set-scroll-restoration!
  "Property.

  Allows web applications to explicitly set default scroll restoration
  on history navigation. This property can be either auto or manual."
  [this val]
  (aset this "scrollRestoration" val))

(defn state
  "Property.

  Returns an any value representing the state at the top of the
  stack. This is a way to look at the state without having to wait
  a popstate event."
  [this]
  (-> this (.-state)))

(defn set-state!
  "Property.

  Returns an any value representing the state at the top of the
  stack. This is a way to look at the state without having to wait
  a popstate event."
  [this val]
  (aset this "state" val))

