(ns leiningen.packr
  (:require
    [clojure.java.io :as io]
    [leiningen.uberjar :as uber]
    [leiningen.core.eval :as ev]
    [clj-http.client :as client])
  (:import [com.badlogicgames.packr Packr Packr$Config Packr$Platform]))

(defn write-file [address filename]
   (with-open [w (clojure.java.io/output-stream filename)]
     (.write w (:body (client/get address {:as :byte-array})))))

(defmacro set-all! [obj m]
    `(do ~@(map (fn [e] `(set! (. ~obj ~(key e)) ~(val e))) m) ~obj))
(defn get-platform [tgt]
  (condp = tgt
    "windows" Packr$Platform/windows
    ;"linux32" Packr$Platform/linux32
    ;"linux64" Packr$Platform/linux
    "linux" Packr$Platform/linux
    "mac"     Packr$Platform/mac
    Packr$Platform/windows))
(defn packr
"Generate and pack an uberjar into an executable with a bundled JRE.

Downloads the latest Packr into ~/.lein-packr-cache/ if it is not present,
and runs it on an uberjar generated by the no-arg uberjar command.

This will by default download 3 JREs (once) from the unofficial-openjdk-builds
project -- Windows x86, Linux (x64 only), and Mac x64.
These will be cached in ~/.lein-packr-cache/ , which will be created if it
doesn't already exist. To download less, pass an extra arg to the `lein packr`
task, one or more of windows, linux, or mac ; only the JREs
corresponding to the given args will be downloaded and bundled."
  [project & args]
  (let [all-targets ["windows" #_"linux32" "linux" "mac"]
        target-actions {
                        "windows" #(when (not (.exists (io/file (System/getProperty "user.home") ".lein-packr-cache/windows.zip")))
                                     (write-file "https://bitbucket.org/alexkasko/openjdk-unofficial-builds/downloads/openjdk-1.7.0-u60-unofficial-windows-i586-image.zip"
                                                 (str (System/getProperty "user.home") "/.lein-packr-cache/windows.zip")))
                        "linux32" #(when (not (.exists (io/file (System/getProperty "user.home") ".lein-packr-cache/linux32.zip")))
                                     (write-file "https://bitbucket.org/alexkasko/openjdk-unofficial-builds/downloads/openjdk-1.7.0-u60-unofficial-linux-i586-image.zip"
                                                 (str (System/getProperty "user.home") "/.lein-packr-cache/linux32.zip")))
                        "linux" #(when (not (.exists (io/file (System/getProperty "user.home") ".lein-packr-cache/linux64.zip")))
                                     (write-file "https://bitbucket.org/alexkasko/openjdk-unofficial-builds/downloads/openjdk-1.7.0-u60-unofficial-linux-amd64-image.zip"
                                                 (str (System/getProperty "user.home") "/.lein-packr-cache/linux64.zip")))
                        "mac"     #(when (not (.exists (io/file (System/getProperty "user.home") ".lein-packr-cache/mac.zip")))
                                     (write-file "https://bitbucket.org/alexkasko/openjdk-unofficial-builds/downloads/openjdk-1.7.0-u60-unofficial-macosx-x86_64-image.zip"
                                                 (str (System/getProperty "user.home") "/.lein-packr-cache/mac.zip")))}
        targets (if (seq args)
                  (filter #(some #{%} all-targets) args)
                  all-targets)
        jar-name (re-find #"[^/\\]+$" (uber/uberjar project))]
    (println "Successfully built uberjar: " jar-name)
    (.mkdirs (io/file (System/getProperty "user.home") ".lein-packr-cache"))
    (client/with-connection-pool {:timeout 20 :threads (count targets) :insecure? false :default-per-route 10}
      (doseq [tgt targets] ((get target-actions tgt "windows"))))
    (doseq [tgt targets] 
      (let [cfg (Packr$Config.)]
        (set-all! cfg
                                   {platform (get-platform tgt)
                                    jdk (str (System/getProperty "user.home") "/.lein-packr-cache/" tgt ".zip")
                                    executable (:name project)
                                    jar  (str "target/uberjar/" jar-name)
                                    mainClass ((comp #(.replace % \. \/) str munge) (:main project))
                                    vmArgs (list "-server" "-Xms512m" "-Xmx512m") ;; "-XX:+TieredCompilation"
                                    minimizeJre 
                                                (into-array ["jre/lib/rt/com/sun/corba" 
                                                             "jre/lib/rt/com/sun/jndi"
                                                             "jre/lib/rt/com/sun/media"
                                                             "jre/lib/rt/com/sun/naming"
                                                             "jre/lib/rt/com/sun/rowset"
                                                             "jre/lib/rt/sun/applet"
                                                             "jre/lib/rt/sun/corba"
                                                             "jre/lib/rt/sun/management"])
                                    resources (if (.exists (io/file "resources")) (list "resources") ())
                                    outDir (str "out-" tgt)})
         (.pack (Packr.) cfg)))))
