(ns kotr.client.stardog.kafka-consumer
  (:require [clojure.algo.generic.functor :refer [fmap]]
            [jackdaw.serdes :as serdes]
            [ksqldb.client.broker-api :as repl]
            [ksqldb.client.ksql-api]
            [ksqldb.client.schema-api]
            [ksqldb.client :as client]
            [ksqldb.client.broker-sedes-config :as cb]
            [clojure.tools.logging :as log]
            [cheshire.core :as json]
            [jackdaw.serdes.resolver :as resolver]))


(defn get-topic-list [context]
  (let [w-set (get context :topics-end-with "_refined")]
    (->> (repl/list-topics context)
         (into [] (filter (fn [m]
                            (reduce (fn [acc v]
                                      (if-let [w (clojure.string/ends-with? m v)]
                                        (reduced w)
                                        acc)
                                      ) false w-set)))))))


(defn get-topic-config [context topic-name t]
  (if (= t :avro)
    (->> (client/invoke context "describe-schema" (str topic-name "-value"))
         (cb/avro-topic-config context topic-name))
    (cb/json-topic-config context topic-name)))


(defn format-data [m]
  (reduce-kv (fn [acc k v]
               (let [k (-> (name k)
                           (clojure.string/lower-case)
                           (clojure.string/replace "-" "_"))]
                 (assoc acc k v))
               ) {} m))


(defn get-default-schema [context topic-name]
  (let [key-identifier (or (get context :topic-id-end-with) "_id")
        topic-identifier-list (get context :topics-end-with ["_refined"])
        t-name (reduce (fn [acc v]
                         (clojure.string/replace acc v "")
                         ) topic-name topic-identifier-list)]
    {:topic-name     t-name
     :topic-key      (str t-name key-identifier)
     :topic-id-end-with key-identifier}))


(defn get-schema [context topic-name]
  (try
    (let [avro-schema (-> (client/invoke context "describe-schema" (str topic-name "-value")))
          key-identifier (or (get context :topic-id-end-with) "_id")

          topic-identifier-list (get context :topics-end-with ["_refined"])
          t-name (reduce (fn [acc v]
                           (clojure.string/replace acc v "")
                           ) topic-name topic-identifier-list)

          type-m (into {} (comp (map (fn [m]
                                       {(clojure.string/lower-case (get m :name))
                                        (second (get m :type))}
                                       ))) (get avro-schema :fields))]
      {:type-m         type-m
       :topic-name     t-name
       :topic-key      (str t-name key-identifier)
       :topic-id-end-with key-identifier})

    (catch Exception e
      (do
        (log/info "could not found any schema name " topic-name)
        nil))))



(defn add-data-listener [context topic-name callback]
  (try
    (let [avro-schema (-> (client/invoke context "describe-schema" (str topic-name "-value"))
                          #_(assoc :key key :name entity_name))
          ui-schema (-> avro-schema (assoc :name topic-name))
          topic-config (cb/avro-topic-config context topic-name avro-schema)]

      (repl/consume context topic-config (fn [data-coll]
                                           (let [w (mapv (fn [{:keys [key value]}]
                                                           (let [v1 (format-data value)
                                                                 ;   v1 (assoc v1 key-name key)
                                                                 ]
                                                             v1
                                                             )
                                                           ) data-coll)]
                                             (when-not (empty? w)
                                               ;(println "---" w)
                                               (callback ui-schema w)))))
      )
    (catch Exception e
      (println "Error in data listener " e))))




(defn add-metadata-listener [context topic-name callback]
  ;(println "--topoic name " topic-name)
  (let [c (cb/edn-topic-config context topic-name)]
    (repl/consume context c (fn [w-coll]
                              ;(println "--" w-coll)
                              (doseq [{:keys [key value]} w-coll]
                                (when (and key value)
                                  ; (log/info "Got new metadata " key)
                                  (callback key value)))))))


(comment
  ;  (take 1 [1 2 3])




  ;  (first (list 2 3 4 ))

  (let [[t-name t-v] (first @schema-atom)
        t-config (build-topic-data-config t-name t-v)]
    ;  (println t-v)

    (repl/consume t-config println))


  (+ 1 1)

  @repl/consume-state-atom




  (def +topic-metadata+
    {"input"
     {:topic-name         "input"
      :partition-count    1
      :replication-factor 1
      :key-serde          {:serde-keyword :jackdaw.serdes.edn/serde}
      :value-serde        {:serde-keyword :jackdaw.serdes.edn/serde}}

     "output"
     {:topic-name         "output"
      :partition-count    1
      :replication-factor 1
      :key-serde          {:serde-keyword :jackdaw.serdes.edn/serde}
      :value-serde        {:serde-keyword :jackdaw.serdes.edn/serde}}})

  (def topic-metadata
    (memoize (fn []
               (fmap #(assoc % :key-serde ((resolver/serde-resolver) (:key-serde %))
                               :value-serde ((resolver/serde-resolver) (:value-serde %)))
                     +topic-metadata+))))

  (repl/list-topics)

  (repl/publish (get (topic-metadata) "input") "the-key" "the-value")

  ;(repl/get-keyvals (get (topic-metadata) "input"))

  (repl/consume (get (topic-metadata) "input") println)

  (reset! repl/consume-state-atom {}))

(comment

  (slurp "module/kotr/src/key-schema.json")
  (slurp "module/kotr/src/value-schema.json")

  (def +topic-metadata+
    {"input1"
     {:topic-name         "input1"
      :partition-count    1
      :replication-factor 1
      :key-serde          {:serde-keyword :jackdaw.serdes.avro.confluent/serde
                           :schema        (slurp "module/kotr/src/key-schema.json")
                           ;:schema-filename "key-schema.json"
                           :key?          true}
      :value-serde        {:serde-keyword :jackdaw.serdes.avro.confluent/serde
                           ;:schema-filename "value-schema.json"
                           :schema        (slurp "module/kotr/src/value-schema.json")
                           :key?          false}}})

  (def serde-resolver
    (partial resolver/serde-resolver :schema-registry-url "http://localhost:8081"))

  (def topic-metadata
    (memoize (fn []
               (fmap #(assoc % :key-serde ((serde-resolver) (:key-serde %))
                               :value-serde ((serde-resolver) (:value-serde %)))
                     +topic-metadata+))))

  (repl/publish (get (topic-metadata) "input1")
                {:the-key "the-key"}
                {:the-value "the-value"})

  (repl/consume (get (topic-metadata) "input1") println)

  (reset! repl/consume-state-atom {})

  ("input" topic-metadata)

  (slurp "value-schema.json"))

(comment

  (slurp "module/kotr/src/key-schema.json")
  (slurp "module/kotr/src/value-schema.json")

  (slurp "data/config_gen/schema/campaign_in.json")

  (def serde-resolver
    (partial resolver/serde-resolver :schema-registry-url "http://localhost:8081"))

  (def schema (-> (slurp "data/config_gen/schema/campaign_in.json")
                  (clojure.string/lower-case)
                  (json/parse-string true)
                  (select-keys [:name :fields])
                  (assoc :type "record")
                  (update :fields (fn [v]
                                    (mapv (fn [m]
                                            (-> (clojure.set/rename-keys m {:schema :type})
                                                ;       (update-in [:type :type ])
                                                )) v)))
                  (json/generate-string)))

  (def +topic-metadata+
    {:topic-name         "CAMPAIGN_IN"
     :partition-count    1
     :replication-factor 1
     :key-serde          (serdes/string-serde)  #_{:serde-keyword :jackdaw.serdes.avro.confluent/serde
                                                   :schema        (slurp "module/kotr/src/key-schema.json")
                                                   ;:schema-filename "key-schema.json"
                                                   :key?          true}
     :value-serde #_(serdes/string-serde)
                         ((serde-resolver)
                           {:serde-keyword :jackdaw.serdes.avro.confluent/serde
                            ;:schema-filename "value-schema.json"
                            :schema        schema
                            :key?          false})})

  (repl/publish +topic-metadata+
                {:the-key "the-key"}
                {:the-value "the-value"})

  (repl/consume +topic-metadata+ #_(get (topic-metadata) "input1") println)

  (reset! repl/consume-state-atom {})

  (slurp "value-schema.json"))


