(ns ksqldb.csv-loader
  (:require [clojure.data.csv :as csv]
            [clojure.java.io :as io]
            [ksqldb.client :as client]
            [clojure.tools.logging :as log]))

(defn as-insert-statement [stream-name header data]
  (let [w (->> (mapv clojure.string/upper-case header)
               (clojure.string/join ", "))
        data (mapv (fn [v] (if (clojure.string/blank? v)
                             "''"
                             (str "'" (clojure.string/replace v "'" "") "'")
                             )) data)
        data (clojure.string/join ", " data)]
    (str "INSERT INTO " stream-name " ( " w " ) VALUES ( " data " ) ;")))


(defn as-insert-statement-batch [stream-name header data-coll]
  (let [w (comp
            (filter not-empty)
            (map (partial as-insert-statement stream-name header)))]
    (->> (into [] w data-coll)
         (clojure.string/join "\n"))))


(defn load-csv-file
  ([csv-file-path callback] (load-csv-file csv-file-path callback {}))
  ([csv-file-path callback m]
   (let [total (get m :total -1)
         chunk-size (get m :chunk-size 100)]
     (with-open [reader (io/reader csv-file-path)]
       (let [[header & data] (csv/read-csv reader :separator \;)]
         (loop [data data
                total-load chunk-size]
           (println "total processing " total-load)
           (callback header (take chunk-size data))
           (when (and (not-empty data)
                      (if (pos? total)
                        (< total-load total)
                        true))
             (recur (drop chunk-size data) (+ total-load chunk-size)))))))))


(defn load-csv-to-kafka
  ([client stream-name file-name] (load-csv-to-kafka client stream-name file-name {}))
  ([client stream-name file-name m]
   (let [xf (fn [header r]
              (when (not-empty r)
                (let [ksql-str (as-insert-statement-batch stream-name header r)]
                  (try
                    (println "Sending to kafka ")
                    (log/info "Sending to kafka ")
                    (client/ksql client ksql-str)
                    (println "Done!! ")
                    (log/info "Done!! ")
                    (catch Exception e
                      (do
                        (println ksql-str)
                        (println "Failed " r)))))))]
     (load-csv-file file-name xf m))))


;INSERT INTO riderLocations (profileId, latitude, longitude) VALUES ('18f4ea86', 37.3903, -122.0643);

(comment




  (load-csv-to-kafka "CUSTOMER_IN" "./data/source/customer.csv" #_{:total 1 :chunk-size 1})


  )






