(ns ksql.ksqldb-api
  (:require [clojure.tools.logging :as log]
            [ksqldb.client :as client]
            [ksql.gen :as gen]
            [ksql.core :as core]
            [ksql.metadata-api :as md]))



(defn ksql
  "Execute ksql statement to KSQL server. KSQL server URL must need to be configured in context  "
  [ksql-str]
  (client/ksql (core/get-context) ksql-str))



(defn get-delta-ksql-statement [context ksql-coll]
  (let [stream-name-coll (->> (client/invoke context "show-streams")
                              (into #{} (comp (map :name))))
        table-name-coll (->> (client/invoke context "show-tables-name")
                             (into #{} #_(comp (map :name))))
        stream-name-coll (into stream-name-coll table-name-coll)
        ;  _ (println stream-name-coll)
        kafka-insert-query (->> (client/invoke context "show-queries")
                                (into #{} (comp (map :queryString)
                                                (remove nil?)
                                                (filter (fn [v]
                                                          ;   (println "-----------" v)
                                                          (clojure.string/starts-with? v "INSERT")))
                                                (map (fn [v] (clojure.string/upper-case v))))))
        xf (comp (remove (fn [v]
                           (let [w (clojure.string/split v #" ")]
                             (if (and (= (first w) "CREATE")
                                      (contains? stream-name-coll (nth w 2)))
                               true false))))
                 (remove (fn [v]
                           (contains? kafka-insert-query (clojure.string/upper-case (str v ";"))))))]
    (into [] xf ksql-coll)))



;; Keep remote separate, good for integration test
(defn ksql-batch [context ksql-coll]
  (->> ksql-coll
       (get-delta-ksql-statement context)
       (run! (partial client/ksql context))))


(comment

  (run! println (list 12 3 4))

  )


(defn invoke-kafka
  "Invoke Kafka API, if there is no invoke API it will throw exception and return list of available API. "
  ([op-name] (client/invoke (core/get-context) op-name))
  ([op-name request]
   (client/invoke (core/get-context) op-name request)))


(defn get-ksqldb-schema
  "Return current schema from Kafka schema registry "
  []
  (md/get-ksqldb-schema (core/get-context)))


(defn remove-ksql-statement [context stream-names with-topics?]
  (let [ksqldb-schema-coll (->> (md/get-ksqldb-schema context)
                                (vals))
        {:keys [ksql topic schema]} (gen/gen-drop-ksql context ksqldb-schema-coll stream-names with-topics?)]
    (run! (partial client/ksql context) ksql)
    (when with-topics?
      (client/invoke context "broker-delete-topics" topic)
      (doseq [schema-name schema]
        (client/invoke context "delete-schema" schema-name)))))



(defn drop-all-stream-and-table
  ([] (drop-all-stream-and-table [] #{"ksql_processing_log"}))
  ([drop-stream-names skip-stream-name-set]
   (let [drop-names (into #{} drop-stream-names)
         drop-stream-names2 (if (empty? drop-names)
                              (->> (invoke-kafka "show-streams")
                                   (mapv :name)
                                   (mapv clojure.string/lower-case))
                              (->> (invoke-kafka "show-streams")
                                   (mapv :name)
                                   (mapv clojure.string/lower-case)
                                   (filter (fn [v] (contains? drop-names v)))))
         drop-table-names (if (empty? drop-names)
                            (->> (invoke-kafka "show-tables")
                                 (mapv :name)
                                 (mapv clojure.string/lower-case))
                            (->> (invoke-kafka "show-tables")
                                 (mapv :name)
                                 (mapv clojure.string/lower-case)
                                 (filter (fn [v] (contains? drop-names v)))))

         m (->> (into drop-stream-names2 drop-table-names)
                (remove (fn [v] (contains? skip-stream-name-set v)))
                (gen/gen-drop-ksql {} (vals (get-ksqldb-schema))))]

     #_(when-not (empty? (get m :ksql))
         (->> (clojure.string/join " " (get m :ksql))
              (ksql))
         (invoke-kafka "broker-delete-topics" (get m :topic [])))
     (doseq [w (invoke-kafka "show-schemas")]
       (invoke-kafka "delete-schema" w)))))


(defn drop-all-stream-and-table-with-topic
  ([] (drop-all-stream-and-table-with-topic [] #{"ksql_processing_log"}))
  ([drop-stream-names skip-stream-name-set]
   (let [drop-names (into #{} drop-stream-names)
         drop-stream-names2 (if (empty? drop-names)
                              (->> (invoke-kafka "show-streams")
                                   (mapv :name)
                                   (mapv clojure.string/lower-case))
                              (->> (invoke-kafka "show-streams")
                                   (mapv :name)
                                   (mapv clojure.string/lower-case)
                                   (filter (fn [v] (contains? drop-names v)))))
         drop-table-names (if (empty? drop-names)
                            (->> (invoke-kafka "show-tables")
                                 (mapv :name)
                                 (mapv clojure.string/lower-case))
                            (->> (invoke-kafka "show-tables")
                                 (mapv :name)
                                 (mapv clojure.string/lower-case)
                                 (filter (fn [v] (contains? drop-names v)))))

         m (->> (into drop-stream-names2 drop-table-names)
                (remove (fn [v] (contains? skip-stream-name-set v)))
                (gen/gen-drop-ksql {} (vals (get-ksqldb-schema))))]

     (when-not (empty? (get m :ksql))
       (->> (clojure.string/join " " (get m :ksql))
            (ksql))
       (invoke-kafka "broker-delete-topics" (get m :topic [])))
     (doseq [w (invoke-kafka "show-schemas")]
       (invoke-kafka "delete-schema" w)))))


(defn delete-all-connector []
  (invoke-kafka "delete-all-connector"))


(defn clean-system []
  (delete-all-connector)
  (drop-all-stream-and-table-with-topic)
  (core/reset-md-repo (core/get-context) (core/new-metadata-repo)))





