(ns ksql.data-api
  (:require [clojure.tools.logging :as log]
            [clojure.core.async :as async]
            [clojure.data.csv :as csv]
            [cheshire.core :as json]
            [ksql.core :as ksql]
            [ksqldb.client :as client]
            [ksqldb.client.broker-api :as bapi]
            [ksql.gen.reader.csv-data-reader :as r]
            [ksqldb.client.broker-sedes-config :as bsc]
            [ksql.gen.protocol :as p]
            [ksql.ksqldb-api :as kapi]
            [ksql.core :as core]
            [ksql.gen-api :as gapi]))


(defn terminate-query [query-id]
  (client/terminate-push-query (client/get-client (core/get-context)) query-id))


(defn- as-query
  ([v total]
   (if (< 1 (count (clojure.string/split v #" ")))
     v
     (str "select * from " v " emit changes" " limit " total " ;")
     ))
  ([v]
   (if (< 1 (count (clojure.string/split v #" ")))
     v
     (str "select * from " v " emit changes;"))))


(defn query-by-duration
  ([^String query-str] (query-by-duration query-str println "latest" 3000))
  ([^String query-str callback] (query-by-duration query-str callback "latest" 3000))
  ([^String query-str callback offset] (query-by-duration query-str callback offset 3000))
  ([^String query-str callback offset duration]
   (client/stream-query-async-with-duration (core/get-context) (as-query query-str) callback offset duration)))


(defn query-by-total
  ([^String query-str total]
   (query-by-total query-str total "earliest"))
  ([^String query-str total offset]
   (client/stream-query (client/get-client (core/get-context)) (as-query query-str total) total offset)))


(comment
  (clojure.string/split "asdf asdf" #" "))



(defn print-query
  ([^String query-str] (print-query query-str "latest" 3000))
  ([^String query-str offset] (print-query query-str offset 3000))
  ([^String query-str offset duration]
   (client/stream-query-async-with-duration (core/get-context) (as-query query-str) println offset duration)))



(defn print-topic
  ([v] (print-topic v "latest" 3000))
  ([v offset] (print-topic v offset 3000))
  ([v offset duration]
   (client/print-topic-by-time (core/get-context) v offset duration println)))


(defn push-data-to-stream
  [strem-name rows]
  (client/push-data-batch (ksql/get-context) strem-name rows))


#_(defn as-entity-name [f-name]
    (let [w (clojure.string/split f-name #"/")]
      (if-let [t (last w)]
        (first (clojure.string/split t #"\."))
        f-name)))


#_(defn as-topic-data [model-coll]
    (doall
      (for [w model-coll]
        (let [[ds_file data_file key entity_name] w
              data_file (as-entity-name data_file)

              v {:ds-name     (as-entity-name ds_file)
                 :entity_name (or data_file entity_name)
                 :key         key}
              k (or entity_name data_file)]
          {:key k :value v}))))


#_(defn push-edn-event+ [context topic-name edn-data-coll]
    (let [c (bsc/edn-topic-config context topic-name)]
      (doseq [{:keys [key value]} edn-data-coll #_(as-topic-data)]
        (bapi/publish context c key value))))


#_(defn push-stardog [context model-coll]
    (log/info "Push stardog kafka context " context)
    (let [w (mapv as-topic-data model-coll)]
      (push-edn-event+ context "_pipeline_metadata" w)
      ))



(declare push-file-to-topic-impl)
(declare push-event-to-topic-impl)


(defn push-edn-event [topic-name edn-data-coll]
  (let [context (core/get-context)
        w (gapi/gen-napping topic-name edn-data-coll)]

    (p/invoke-dataflow-impl "create-dataflow" context w)
    (push-event-to-topic-impl topic-name edn-data-coll)))


(defn push-json-event [topic-name json-str]
  (push-edn-event topic-name (json/parse-string json-str true)))


(defn push-file [topic-name file-path]
  (let [context (core/get-context)
        mapping (gapi/gen-mapping-by-file topic-name file-path)]
    ;  (clojure.pprint/pprint mapping)
    (p/invoke-dataflow-impl "create-dataflow" context mapping)
    (push-file-to-topic-impl topic-name file-path)))


(defn push-meta-model [directory-name]
  (let [context (core/get-context)
        v ["_" "_" (str "(from_csv_folder \"" directory-name "\" )")
           "_" "_" "(value_format 'json' )" ";"
           "_" "_" (str "(from_json_folder \"" directory-name "\" )" ";")
           "_" "_" (str "(from_xml_folder \"" directory-name "\" )" ";")]]
    (p/invoke-dataflow-impl "create-dataflow" context v)))


(defn push-directory [directory-name]
  (let [csv-file-list (r/get-all-file-name directory-name "glob:*.{csv}")
        json-file-list (r/get-all-file-name directory-name "glob:*.{json}")
        xml-file-list (r/get-all-file-name directory-name "glob:*.{xml}")]
    ;  (import-meta-model context directory-name)
    (doseq [[file-name file-path] csv-file-list]
      (push-file file-name file-path))
    (doseq [[file-name file-path] json-file-list]
      (push-file file-name file-path))
    (doseq [[file-name file-path] xml-file-list]
      (push-file file-name file-path))))


;;;;;;;;;;;;;;;;;;;;;;;;;; push impl ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defn push-event-to-topic-impl [topic-name edn-data-coll]
  (let [context (core/get-context)
        topic-config (bsc/json-topic-config context topic-name)
        input-chan (bapi/publish-async context topic-config)]
    (doseq [v edn-data-coll]
      (let [w (json/generate-string v)
            w {:value w}]
        (async/>!! input-chan w)))
    (log/info "total msg send to kafka " (count edn-data-coll))
    (async/>!! input-chan "done!")))


(defmulti push-file-to-topic-impl (fn [_ file-path]
                                    ;   (println "--" file-path)
                                    (let [w (last (clojure.string/split file-path #"\."))]
                                      (clojure.string/lower-case w))))


(defmethod push-file-to-topic-impl :default
  [topic-name file-path]
  (let [w (last (clojure.string/split file-path #"\."))]
    (throw (Exception. "now support only json, xml and csv file " {:provided w
                                                                   :expected #{"json" "xml" "csv"}}))))


(defmethod push-file-to-topic-impl "json"
  [topic-name json-file-path]
  (log/info " ---------start push file---------- " json-file-path)
  (let [context (core/get-context)
        topic-config (bsc/json-topic-config context topic-name)
        input-chan (bapi/publish-async context topic-config)
        total (atom 0)]
    (try
      (with-open [rdr (clojure.java.io/reader json-file-path)]
        (doseq [line (line-seq rdr)]
          (async/>!! input-chan {:value line})
          (swap! total inc))
        (async/>!! input-chan "done!"))
      (catch Exception e
        (do
          (async/>!! input-chan "done!")
          (ex-data e))))
    (log/info "--------- End push file to kafka with " {:file-path json-file-path :total @total})))



(defmethod push-file-to-topic-impl "csv"
  [topic-name csv-file-path]
  (log/info " ---------start push file---------- " csv-file-path)
  (let [context (core/get-context)
        topic-config (bsc/json-topic-config context topic-name)
        input-chan (bapi/publish-async context topic-config)
        total (atom 0)
        separator (r/get-separator csv-file-path)
        header (r/read-csv-header (clojure.string/replace csv-file-path "'" ""))
        header (r/format-csv-header header)]

    (try
      (with-open [rdr (csv/read-csv (clojure.java.io/reader csv-file-path) :separator separator)  #_(clojure.java.io/reader csv-file-path)]
        (doseq [line (rest rdr)]
          (let [w (zipmap header line)
                w-json (json/generate-string w)]
            (async/>!! input-chan {:value w-json})
            (swap! total inc)))
        (async/>!! input-chan "done!"))
      (catch Exception e
        (do
          (async/>!! input-chan "done!")
          (ex-data e))))
    (log/info "--------- End push file to kafka with " {:file-path csv-file-path :total @total})))



(defmethod push-file-to-topic-impl "xml"
  [topic-name xml-file-path]
  (let [context (core/get-context)
        topic-config (bsc/json-topic-config context topic-name)
        input-chan (bapi/publish-async context topic-config)
        total (atom 0)
        ]

    (try
      (let [w (r/convert-xml-file-to-edn xml-file-path false)  #_(-> rdr
                                                                     xml/parse
                                                                     (xmltojson/parse)
                                                                     #_(json/generate-string))
            w1 (json/generate-string w)
            ]
        ; (clojure.pprint/pprint w)
        (async/>!! input-chan {:value w1})
        (swap! total inc)
        (async/>!! input-chan "done!")
        )
      #_(with-open [rdr (clojure.java.io/file xml-file-path)]

          (async/>!! input-chan "done!"))
      (catch Exception e
        (do
          (async/>!! input-chan "done!")
          (ex-data e))))
    (log/info "--------- End push file to kafka with " {:file-path xml-file-path :total @total})))



