(ns ksql.ksqldb-api
  (:require [clojure.core.async :as async]
            [clojure.data.csv :as csv]
            [clojure.tools.logging :as log]
            [cheshire.core :as json]
            [ksqldb.client.broker-api :as bapi]
            [ksqldb.client.broker-sedes-config :as bsc]
            [ksqldb.client :as client]
            [ksql.gen.reader.csv-data-reader :as r]
            [ksql.gen :as gen]
            [ksql.core :as core]
            [ksql.metadata-api :as md]))





(defn ksql
  "Execute ksql statement to KSQL server. KSQL server URL must need to be configured in context  "
  [ksql-str]
  (client/ksql (core/get-context) ksql-str))


(defn invoke-kafka
  "Invoke Kafka API, if there is no invoke API it will throw exception and return list of available API. "
  ([op-name] (client/invoke (core/get-context) op-name #_{:op op-name}))
  ([op-name request]
   (client/invoke (core/get-context) op-name request #_{:op op-name :request request})))

(defn get-ksqldb-schema
  "Return current schema from Kafka schema registry "
  []
  (md/get-ksqldb-schema (core/get-context)))


(defn drop-all-stream
  ([] (drop-all-stream #{"ksql_processing_log"}))
  ([skip-stream-name-set]
   (let [m (->> (invoke-kafka "show-streams")
                (mapv :name)
                (mapv clojure.string/lower-case)
                (remove (fn [v] (contains? skip-stream-name-set v)))
                (gen/gen-drop-ksql {} (vals (get-ksqldb-schema))))
         ]
     ;(println "-------------" m )
     (when-not (empty? (get m :ksql))
       (->> (clojure.string/join " " (get m :ksql))
            (ksql)
            )
       )
     )))


(defn drop-all-stream-and-table-with-topic
  ([] (drop-all-stream-and-table-with-topic [] #{"ksql_processing_log"}))
  ([drop-stream-names skip-stream-name-set]
   (let [drop-names (into #{} drop-stream-names)
         drop-stream-names2 (if (empty? drop-names)
                              (->> (invoke-kafka "show-streams")
                                   (mapv :name)
                                   (mapv clojure.string/lower-case))
                              (->> (invoke-kafka "show-streams")
                                   (mapv :name)
                                   (mapv clojure.string/lower-case)
                                   (filter (fn [v] (contains? drop-names v)))))
         drop-table-names (if (empty? drop-names)
                            (->> (invoke-kafka "show-tables")
                                 (mapv :name)
                                 (mapv clojure.string/lower-case))
                            (->> (invoke-kafka "show-tables")
                                 (mapv :name)
                                 (mapv clojure.string/lower-case)
                                 (filter (fn [v] (contains? drop-names v)))))

         m (->> (into drop-stream-names2 drop-table-names)
                (remove (fn [v] (contains? skip-stream-name-set v)))
                (gen/gen-drop-ksql {} (vals (get-ksqldb-schema))))]

     (when-not (empty? (get m :ksql))
       (->> (clojure.string/join " " (get m :ksql))
            (ksql))
       (invoke-kafka "broker-delete-topics" (get m :topic []))))))


(defn delete-all-connector []
  (invoke-kafka "delete-all-connector"))


(defn push-data-to-topic [topic-name edn-data-coll]
  (let [context (core/get-context)
        topic-config (bsc/json-topic-config context topic-name)
        input-chan (bapi/publish-async context topic-config)]
    (doseq [v edn-data-coll]
      (let [w (json/generate-string v)
            w {:value w}]
        (async/>!! input-chan w)))
    (log/info "total msg send to kafka " (count edn-data-coll))
    (async/>!! input-chan "done!")))



(defmulti push-file-to-topic (fn [_ file-path]
                               ;   (println "--" file-path)
                               (let [w (last (clojure.string/split file-path #"\."))]
                                 (clojure.string/lower-case w)
                                 )

                               ))


(defmethod push-file-to-topic :default
  [topic-name file-path]
  (let [w (last (clojure.string/split file-path #"\."))]
    (throw (Exception. "now support only json, xml and csv file " {:provided w
                                                                   :expected #{"json" "xml" "csv"}
                                                                   }))
    )
  )


(defmethod push-file-to-topic "json"
  [topic-name json-file-path]
  (log/info " ---------start push file---------- " json-file-path)
  (let [context (core/get-context)
        topic-config (bsc/json-topic-config context topic-name)
        input-chan (bapi/publish-async context topic-config)
        total (atom 0)]
    (try
      (with-open [rdr (clojure.java.io/reader json-file-path)]
        (doseq [line (line-seq rdr)]
          (async/>!! input-chan {:value line})
          (swap! total inc))
        (async/>!! input-chan "done!"))
      (catch Exception e
        (do
          (async/>!! input-chan "done!")
          (ex-data e))))
    (log/info "--------- End push file to kafka with " {:file-path json-file-path :total @total})))



(defmethod push-file-to-topic "csv"
  [topic-name csv-file-path]
  (log/info " ---------start push file---------- " csv-file-path)
  (let [context (core/get-context)
        topic-config (bsc/json-topic-config context topic-name)
        input-chan (bapi/publish-async context topic-config)
        total (atom 0)
        separator (r/get-separator csv-file-path)
        header (r/read-csv-header (clojure.string/replace csv-file-path "'" ""))
        header (r/format-csv-header header)]

    (try
      (with-open [rdr (csv/read-csv (clojure.java.io/reader csv-file-path) :separator separator)  #_(clojure.java.io/reader csv-file-path)]
        (doseq [line (rest rdr)]
          (let [w (zipmap header line)
                w-json (json/generate-string w)]
            (async/>!! input-chan {:value w-json})
            (swap! total inc)))
        (async/>!! input-chan "done!"))
      (catch Exception e
        (do
          (async/>!! input-chan "done!")
          (ex-data e))))
    (log/info "--------- End push file to kafka with " {:file-path csv-file-path :total @total})))



(defmethod push-file-to-topic "xml"
  [topic-name xml-file-path]
  (let [context (core/get-context)
        topic-config (bsc/json-topic-config context topic-name)
        input-chan (bapi/publish-async context topic-config)
        total (atom 0)
        ]

    (try
      (let [w (r/convert-xml-file-to-edn xml-file-path false)  #_(-> rdr
                                                                     xml/parse
                                                                     (xmltojson/parse)
                                                                     #_(json/generate-string))
            w1 (json/generate-string w)
            ]
        ; (clojure.pprint/pprint w)
        (async/>!! input-chan {:value w1})
        (swap! total inc)
        (async/>!! input-chan "done!")
        )
      #_(with-open [rdr (clojure.java.io/file xml-file-path)]

          (async/>!! input-chan "done!"))
      (catch Exception e
        (do
          (async/>!! input-chan "done!")
          (ex-data e))))
    (log/info "--------- End push file to kafka with " {:file-path xml-file-path :total @total})))

