(ns ksqldb.client.broker-api
  (:require [cheshire.core :as json]
            [clj-http.client :as http]
            [clojure.core.async :as async]
            [ksqldb.client.protocol :as c]
            [jackdaw.admin :as ja]
            [jackdaw.client :as jc]
            [jackdaw.client.log :as jcl]
            [clojure.tools.logging :as log]))




(defn publish
  "Takes a topic config and record value, and (optionally) a key and
  parition number, and produces to a Kafka topic."
  ([context topic-config value]
   (with-open [client (jc/producer {"bootstrap.servers" (get context :kafka-broker)} topic-config)]
     @(jc/produce! client topic-config value))
   nil)

  ([context topic-config key value]
   (with-open [client (jc/producer {"bootstrap.servers" (get context :kafka-broker)} topic-config)]
     @(jc/produce! client topic-config key value))
   nil)

  ([context topic-config partition key value]
   (with-open [client (jc/producer {"bootstrap.servers" (get context :kafka-broker)} topic-config)]
     @(jc/produce! client topic-config partition key value))
   nil))



(defn publish-async
  "Takes a topic config and record value, and (optionally) a key and
  parition number, and produces to a Kafka topic. Return async chan"
  ([context topic-config]
   (let [c (async/chan 1)
         status-msg "done!"
         init-broker-status (promise)
         _ (future
             (try
               (with-open [client (jc/producer {"bootstrap.servers" (get context :kafka-broker)} topic-config)]
                 (deliver init-broker-status status-msg)
                 (loop [m (async/<!! c)]
                   (if (= m status-msg)
                     (do
                       (log/info "publish message done for " topic-config)
                       status-msg)
                     (let [v (get m :value)
                           k (get m :key)
                           partition (get m :partition)]
                       @(jc/produce! client topic-config v)
                       (recur (async/<!! c))))))
               (catch Exception e
                 (do
                   (log/error (str "Kafka connection error for " (get context :kafka-broker) )  e)
                   #_(log/info "error init kafka connection " (ex-data e))
                   (deliver init-broker-status (ex-data e))))))
         v @init-broker-status]
     (if (= v status-msg)
       c
       (throw (Exception. (pr-str v)))))))


;;; ------------------------------------------------------------
;;;
;;; Produce and consume records
;;;


(defn get-records
  "Takes a topic config, consumes from a Kafka topic, and returns a
  seq of maps."
  ([client topic-config]
   (get-records client topic-config 200))

  ([client topic-config polling-interval-ms]
   (let [consumer-config {"bootstrap.servers"  (get client :kafka-broker)
                          "auto.offset.reset"  "earliest"
                          "enable.auto.commit" "false"}
         client-config (assoc consumer-config
                         "group.id"
                         (str (java.util.UUID/randomUUID)))]
     (with-open [client (jc/subscribed-consumer client-config [topic-config])]
       (doall (jcl/log client 100 seq))))))

(defn get-keyvals
  "Takes a topic config, consumes from a Kafka topic, and returns a
  seq of key-value pairs."
  ([client topic-config]
   (get-keyvals client topic-config 20))

  ([client topic-config polling-interval-ms]
   (map (juxt :key :value) (get-records client topic-config polling-interval-ms))))


(defn list-topics
  "Returns a list of Kafka topics."
  [client]
  (with-open [client (ja/->AdminClient {"bootstrap.servers" (get client :kafka-broker)})]
    (mapv :topic-name (ja/list-topics client))))








(defn topic-exists?
  "Takes a topic name and returns true if the topic exists."
  [client topic-config]
  (with-open [client (ja/->AdminClient {"bootstrap.servers" (get client :kafka-broker)})]
    (ja/topic-exists? client topic-config)))


(defn delete-topics
  [client topics-name]
  (with-open [client (ja/->AdminClient {"bootstrap.servers" (get client :kafka-broker)})]
    #_(ja/topic-exists? client topic-config)
    (ja/delete-topics! client topics-name)))




(defmethod c/invoke "broker-describe-topic"
  [client {:keys [request]}]
  (when (nil? request)
    (throw (ex-info "Invalid request, topic name is null please provide topic name " {:topic-name request})))
  (with-open [client (ja/->AdminClient {"bootstrap.servers" (get client :kafka-broker)})]
    (-> (ja/describe-topics client)
        (get request))))


(defmethod c/invoke "broker-create-topic"
  [client {:keys [request]}]
  #_(when (nil? request)
      (throw (ex-info "Invalid request, topic name is null please provide topic name " {:topic-name request})))
  (with-open [client (ja/->AdminClient {"bootstrap.servers" (get client :kafka-broker)})]
    (ja/create-topics! client request)
    #_(let [(hash-map :topic-name v)]

        )
    ))


#_(defmethod c/invoke "broker-count-msg"
    [client {:keys [request]}]
    #_(when (nil? request)
        (throw (ex-info "Invalid request, topic name is null please provide topic name " {:topic-name request})))
    (with-open [client (ja/->AdminClient {"bootstrap.servers" (get client :kafka-broker)})]
      (jcl/count-messages client {:topic-name request})
      #_(let [(hash-map :topic-name v)]
          (ja/create-topics! client)
          )
      ))



(def consume-state-atom (atom {}))

(defn consume
  [client topic-config callback]
  (log/info "start consumeing new topic " (get topic-config :topic-name))
  (let [topic-name (get topic-config :topic-name)
        consumer-config {"bootstrap.servers"  (get client :kafka-broker)
                         "auto.offset.reset"  "earliest"
                         "enable.auto.commit" "false"
                         "group.id"           (str (java.util.UUID/randomUUID))}
        list-topic-set (into #{} (list-topics client))]

    (when-not (get-in @consume-state-atom [topic-name])
      (log/info "running consumer " @consume-state-atom)
      (log/info "Kafka broker" (get client :kafka-broker))
      (swap! consume-state-atom (fn [m] (assoc m topic-name true)))
      (future
        (let [consumer (jackdaw.client/consumer consumer-config topic-config)]
          (with-open [client (jackdaw.client/subscribe consumer [topic-config])]
            (jackdaw.client/seek-to-beginning-eager client)
            (loop []
              (do
                (->> (jackdaw.client/poll client 1000)
                     (map #(select-keys % [:key :value]))
                     (callback))
                (when (get-in @consume-state-atom [topic-name])
                  (recur)))))
          (log/info "Close consumer"))))))



(defmethod c/invoke "broker-delete-topics"
  [client {:keys [request]}]
  (when-not (sequential? request)
    (throw (ex-info "please provide list of topics " {:provided request})))
  (delete-topics client (into [] (map (fn [v]
                                        ;(println "--topic name fopr delete" v)
                                        (hash-map :topic-name v)
                                        )) request)))



(defmethod c/invoke "broker-topic-exists"
  [client {:keys [request]}]
  (topic-exists? client {:topic-name request}))


(defmethod c/invoke "broker-publish-msg"
  [client {:keys [request]}]
  (let [{:keys [data topic-config]} request]
    (doseq [[k v] data]
      (publish client topic-config k v))))


(defmethod c/invoke "broker-consume-msg"
  [client {:keys [request]}]
  (let [{:keys [callback topic-config]} request]
    (consume client topic-config callback)
    #_(doseq [[k v] data]
        (publish client topic-config k v))))


(defmethod c/invoke "broker-stop-consume-msg"
  [client {:keys [request]}]
  (swap! consume-state-atom dissoc request)
  #_(let [{:keys [topic-name]} request]

      #_(consume client topic-config callback)
      #_(doseq [[k v] data]
          (publish client topic-config k v))))




(comment


  (use '[ksql.client.broker-sedes-config])

  (use '[ksql.client])

  (jcl/count-messages)

  ;(topic-exists? context {:topic-name "test1"})

  (let [topic-name "test1"]
    (c/invoke context {:op "broker-publish-msg" :request {:topic-config (edn-topic-config topic-name)
                                                          :data         [["1" {:lname "hello"}]
                                                                         ["2" {:lname "hello"}]]
                                                          }})
    )


  (c/invoke context {:op "broker-count-msg" :request "ref_mapping"})


  (let [topic-name "ref_mapping"]
    (c/invoke context {:op "broker-consume-msg" :request {:topic-config (avro-topic-config context topic-name)
                                                          :callback     (fn [v]
                                                                          (println "" v)
                                                                          )
                                                          }})
    )


  (let [schema {:type         "record",
                :name         "ref_mapping",
                :namespace    "io.datafy.connector.kafka.ksql",
                :fields       [{:name "global_value", :type ["null" "string"], :default nil}
                               {:name "column_value", :type ["null" "string"], :default nil}
                               {:name "column_name", :type ["null" "string"], :default nil}
                               {:name "oe_name", :type ["null" "string"], :default nil}
                               {:name "data_category", :type ["null" "string"], :default nil}],
                :connect.name "io.datafy.connector.kafka.ksql.ref_mapping"}]
    (c/invoke context {:op      "broker-create-topic"
                       :request [
                                 (avro-topic-config context "test8" schema)
                                 ]
                       })
    )

  (list-topics context)
  ;(reset! consume-state-atom {})

  (c/invoke context {:op "broker-stop-consume-msg" :request "ref_mapping"})

  (c/invoke context {:op "describe-topic" :request "ref_mapping"})

  (c/invoke context {:op "describe-schema" :request "test8-value"})

  (list-topics context)

  (c/invoke context {:op "broker-delete-topics" :request ["test1"]})


  (c/invoke context {:op "broker-topic-exists" :request "test1"})

  ;  (topic-config-for-metadata "test1")

  )