(ns ksql.gen.file-util
  (:require [cheshire.core :as json]
            [clojure.java.io :as io]
            [clojure.data.csv :as csv]
            [clojure.tools.logging :as log]))


(defn csv-data->maps [csv-data]
  (map zipmap
       (->> (first csv-data)                                ;; First row is the header
            (map keyword)                                   ;; Drop if you want string keys instead
            repeat)
       (rest csv-data)))

(defn read-csv-file [file-name]
  ; (log/info "reading csv file " file-name)
  (with-open [reader (io/reader file-name)]
    (->> (csv/read-csv reader :separator \;)
         (csv-data->maps)
         doall)))


(defn get-file-list [ext dir-name]
  (into [] (comp (filter #(.isFile %))
                 (map #(.getAbsolutePath %))
                 (filter #(clojure.string/ends-with? % ext))
                 ) (file-seq (clojure.java.io/file dir-name))))


(defn get-csv-header [separator file-name]
  (with-open [reader (io/reader file-name)]
    (first
      (csv/read-csv reader :separator separator))))





(defn get-file-list-by-prefix [mapping-prefix mapping-folder]
  (->> (get-file-list ".csv" mapping-folder)
       (filter (fn [v]
                 (clojure.string/starts-with? (last (clojure.string/split v #"/"))
                                              mapping-prefix)))))


(defn write-to-file [file-name coll]
  (let [header (into [] (keys (first coll)))
        rows (into [] (distinct (mapv (apply juxt header) coll)))
        header (mapv name header)
        rows (into [header] rows)]
    (with-open [writer (io/writer file-name)]
      (csv/write-csv writer
                     rows
                     :separator \;))))



(defn save-mapping-as-csv [file-name mapping-coll-of-coll]
  (let [
        file-exits? (.exists (io/as-file file-name))]
    (with-open [writer (io/writer file-name :append true)]
      (if file-exits?
        (csv/write-csv writer
                       [[]]
                       :separator \;)
        (let [header ["name" "field_name" "transfer_fn" "type" "doc"]]
          (csv/write-csv writer
                         [header]
                         :separator \;)

          )

        )


      (doall (for [mapping-coll mapping-coll-of-coll]
               (do
               ;  (println "--" mapping)
                 (csv/write-csv writer
                                mapping-coll
                                :separator \;)
                (csv/write-csv writer
                              []
                              :separator \;)
                 )

               ) )

      )

    )


  )



;;;;;;;;;;;;;;;;;;;;;;;;;, Save model ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defn save-ksql-to-run-sh [context folder-name f-name]
  (let [ksql-url (get-in context [:ksql-url])]
    (spit (str folder-name "/run.sh")
          (str "curl -X POST  -H \"Content-Type: application/vnd.ksql.v1+json; charset=utf-8\" --data @" f-name "  " ksql-url "/ksql\n")
          :append true)))

(defn save-ksql-as-rest-req [context folder-name ksql-coll]
  (let [ksql-str (clojure.string/join " " ksql-coll)
        w (-> (hash-map :streamsProperties {:ksql.streams.auto.offset.reset "earliest"}
                        :ksql ksql-str)
              (json/generate-string {:pretty true}))
        file-name "ksql.json"
        file-name2 (str folder-name "/" file-name)]
    (io/make-parents file-name2)
    (spit file-name2 w)
    (save-ksql-to-run-sh context folder-name file-name)))


(defn save-to-run-sh [context folder-name f-name]
  (let [conn-url (get-in context [:connector-url])]
    (spit (str folder-name "/run.sh")
          (str "curl -X POST -H \"Accept:application/json\" -H \"Content-Type: application/json\"  --data @" f-name " " conn-url "\n")
          :append true)))

(defn save-connector [context folder-name connector-batch]
  (doseq [m connector-batch]
    (let [; m-v (json/parse-string m true)
          f-name (str (clojure.string/lower-case (get m :name)) ".json")
          path (str folder-name "/" f-name)]
      (io/make-parents path)
      (spit path (json/generate-string m {:pretty true}))
      (save-to-run-sh context folder-name f-name))))



(defn save-arvo-model [folder-name model-coll]

  (let [folder-name (str folder-name "/schema")]
    (doseq [m model-coll]
      (let [t-name (get m :name)
            path (str folder-name "/" t-name ".json")
            fields (get m :fields)
            m (-> m (dissoc :fields) (assoc :fields fields))]
        (io/make-parents path)
        (spit path (json/generate-string m {:pretty true}))))))


(defn save-ksql [folder-name ksql-coll]
  (let [m (clojure.string/join "\n" ksql-coll)
        path (str folder-name "/ksql.ksql")]
    (io/make-parents path)
    (spit path m :append true)))


(defn persist-data-flow [app-context model-context]
  (let [gen-config-dir (get app-context :gen-config-dir)]
    (save-arvo-model gen-config-dir (get model-context :schema-coll))
    (save-connector app-context (str gen-config-dir "/kafka") (get model-context :source-connector))
    (save-connector app-context (str gen-config-dir "/kafka") (get model-context :sink-connector))
    (save-ksql (str gen-config-dir "/kafka") (get model-context :ksql-coll))
    (save-ksql-as-rest-req app-context (str gen-config-dir "/kafka") (get model-context :ksql-coll))))

