(ns ksql.ingestion-api
  (:require [ksql.dataflow-impl :as di]
           [ksql.gen.reader.csv-data-reader :as r]
           [ksql.gen.protocol :as p]
           [ksql.ksqldb-api :as kapi]
           [cheshire.core :as json]
           [ksql.gen-api :as gapi]
           [ksql.core :as core]))


(defn invoke-dataflow
  [op-name request]
  (p/invoke-dataflow op-name (core/get-context) request))


(defn reset-schema-m []
  (update (core/get-context) :mapping-schema (fn [v-atom]
                                               (reset! v-atom []))))



(defn export-ksql-to-file
  "Export generated KSQL statement to file
  file-name: Saving file name
  mapping: Mapping file or mapping as data
  "
  [file-name mapping]
  (let [o (gapi/gen-ksql-by-mapping mapping)
        o (clojure.string/join "\n" o)]
    (spit file-name o :append true)))


(defn import-ksql-from-file
  "Import KSQl to KSQl server after apply delta "
  [file-name]
  (->> (clojure.string/split (slurp file-name) #"\n")
       (remove (fn [v] (clojure.string/blank? v)))
       (vec)
       (di/push-ksql-statement (core/get-context))))



(defn push-edn-event [topic-name edn-data-coll]
  (let [context (core/get-context)
        w (gapi/gen-napping topic-name edn-data-coll)]

    (p/invoke-dataflow "create-dataflow" context w)
    (kapi/push-data-to-topic topic-name edn-data-coll)))


(defn push-json-event [topic-name json-str]
  (push-edn-event topic-name (json/parse-string json-str true)))



(defn push-file [topic-name file-path]
  (let [context (core/get-context)
        mapping (gapi/gen-mapping-by-file topic-name file-path)]
  ;  (clojure.pprint/pprint mapping)
    (p/invoke-dataflow "create-dataflow" context mapping)
    (kapi/push-file-to-topic topic-name file-path)))


(defn push-meta-model [directory-name]
  (let [context (core/get-context)
        v ["_" "_" (str "(from_csv_folder \"" directory-name "\" )")
           "_" "_" "(value_format 'json' )" ";"
           "_" "_" (str "(from_json_folder \"" directory-name "\" )" ";")
           "_" "_" (str "(from_xml_folder \"" directory-name "\" )" ";")]]
    (p/invoke-dataflow "create-dataflow" context v)))


(defn push-directory [directory-name]
  (let [csv-file-list (r/get-all-file-name directory-name "glob:*.{csv}")
        json-file-list (r/get-all-file-name directory-name "glob:*.{json}")
        xml-file-list (r/get-all-file-name directory-name "glob:*.{xml}")]
    ;  (import-meta-model context directory-name)
    (doseq [[file-name file-path] csv-file-list]
      (push-file file-name file-path))
    (doseq [[file-name file-path] json-file-list]
      (push-file file-name file-path))
    (doseq [[file-name file-path] xml-file-list]
      (push-file file-name file-path))))

