(ns utils.rethink-service
  (:require [rethinkdb.query :as r]
            [utils.logger :as logger]))


(defn connect [db-name] (r/connect :host (System/getenv "RDBHOST") :port (System/getenv "RDBPORT") :db db-name))

(defn find-by-id [id table-name db-name]
  (with-open [conn (connect db-name)]
    (->
      (r/table table-name)
      (r/get id)
      (r/run conn))))

(defn find-by-filter [filter table-name db-name]
  (with-open [conn (connect db-name)]
    (->
      (r/table table-name)
      (r/filter filter)
      (r/run conn))))

(defn delete-by-filter [filter table-name db-name]
  (with-open [conn (connect db-name)]
    (->
      (r/table table-name)
      (r/filter filter)
      (r/delete)
      (r/run conn))))

(defn delete-by-id [db-name table-name id]
  (with-open [conn (connect db-name)]
    (->
      (r/table table-name)
      (r/get id)
      (r/delete)
      (r/run conn))))

(defn find-by-index [value index table-name db-name]
  (with-open [conn (connect db-name)]
    (->
      (r/table table-name)
      (r/get-all [value] {:index index})
      (r/run conn))))


(defn find-by-index-coerce-array [value index table-name db-name]
  (with-open [conn (connect db-name)]
    (->
      (r/table table-name)
      (r/get-all [value] {:index index})
      (r/coerce-to "array")
      (r/run conn))))

(defn find-all [table-name db-name]
  (with-open [conn (connect db-name)]
    (->
      (r/table table-name)
      (r/run conn))))

(defn find-all-coerce-array [table-name db-name]
  (with-open [conn (connect db-name)]
    (->
      (r/table table-name)
      (r/coerce-to "array")
      (r/run conn))))

(defn replace-doc [db-name table-name message]
  (with-open [conn (connect db-name)]
    (->
      (r/table table-name)
      (r/insert message {:conflict "replace"})
      (r/run conn))))

(defn replace-one [db table pk doc]
  (replace-doc db table (merge doc {:id pk})))

(defn update-doc [db-name table-name message]
  (with-open [conn (connect db-name)]
    (->
      (r/table table-name)
      (r/insert message {:conflict "update"})
      (r/run conn))))

(defn- retrieve-created [pks db table]
  (let [result (map #(find-by-id % table db) pks)]
    (if (= 1 (count result))
      (first result)
      (seq result))))

(defn insert [db table doc]
  (with-open [conn (connect db)]
    (-> (r/table table)
        (r/insert doc {:conflict "error"})
        (r/run conn)
        :generated_keys
        (retrieve-created db table))))

(defn update [db table pk doc]
  (with-open [conn (connect db)]
    (-> (r/table table)
        (r/insert (merge doc {:id pk}) {:conflict "update"})
        (r/run conn)
        :generated_keys
        (retrieve-created db table))))

(defn update-one [db table pk doc]
  (with-open [conn (connect db)]
    (-> (r/table table)
        (r/insert (merge doc {:id pk}) {:conflict "update"})
        (r/run conn))
    (retrieve-created [pk] db table)))
