(ns utils.health-service
  (:require [utils.mysql-service :as mysql]
            [rethinkdb.core :as rethink]
            [langohr.core :as rabbit]
            [utils.rabbit-service :as rabbit-service]
            [clojure.java.io :as io]
            [clj-http.client :as http]
            [utils.logger :as logger])
  (:import java.io.File))

(def elasticsearch-host (System/getenv "ESHOST"))

(defn check-mysql-connection []
  (let [test-query "select 1 + 1 AS result"]
    (try (mysql/run-test-read-replica test-query)
         (mysql/run-test-write-master test-query)
         {:status "UP"
          :database "MySql"}
         (catch Exception e
           (logger/err (.getMessage e))
           {:status "DOWN"
            :database "MySql"}))))

(defn check-rethink-connection []
  (try (with-open [c (rethink/connect :host (System/getenv "RDBHOST") :port 28015 :db-name nil)]
          {:status "UP"})
       (catch Exception e
         (logger/err (.getMessage e))
         {:status "DOWN"})))

(defn check-rabbit-connection []
  (try 
      (with-open [c (rabbit/connect (rabbit-service/connection-settings "AMQPHOST" "AMQPUSER" "AMQPPASS"))]
          {:status "UP"})
       (catch Exception e
         (logger/err (.getMessage e))
         {:status "DOWN"})))

(defn check-diskspace-status
  ([total free]
   (check-diskspace-status total free 0.90))
  ([total free threshold]
   (if (<= (float (/ (- total free) total)) threshold)
     "UP"
     "DOWN")))

(defn get-disk-space [threshold]                            ;threshold as a percent of total
  (let [file (io/file "/")
        total (.getTotalSpace file)
        free (.getFreeSpace file)]
    {:status (check-diskspace-status total free threshold)
     :total  total
     :free   free}))

(defn get-environment []
  (System/getenv "ENVIRONMENT"))

(defn elastic-search-request []
  (let [env (get-environment)]
    (http/get (str elasticsearch-host "/_cluster/health/")
              {:client-params    {:cookie-policy (constantly nil)}
               :content-type     :json
               :throw-exceptions false
               :as               :json
               :insecure?        true})))

(defn get-elastic-search-connection []
  (if (= (:status (elastic-search-request)) 200)
    {:status "UP"}
    {:status "DOWN"}))

(defn wrap-health-result [type key name f]
  {
    :type type
    :key key 
    :status (f)
  })


(def default-healths #{:mysql :rethink :rabbit :diskSpace :elasticSearch})

(def health-map 
  {:db (partial wrap-health-result :db :db "mysql" check-mysql-connection)
   :rethink (partial wrap-health-result :db :rethink "rethink" check-rethink-connection)
   :rabbit (partial wrap-health-result :top :rabbit "rabbit" check-rabbit-connection)
   :diskSpace (partial wrap-health-result :top :diskSpace "diskSpace" (partial get-disk-space 0.90))
   :elasticSearch (partial wrap-health-result :top :elasticSearch "elasticSearch" get-elastic-search-connection)
   })


(defn maybe-check-health 
  [keyset main-key obj]
  (if (or
        (empty? keyset)
        (contains? keyset main-key))
    (let [result-fn (main-key health-map)
          result (result-fn)]
      (assoc obj (:key result) (:status result)))
    obj))


(defn reduce-helper [r v]
  (if (= (:status v) "UP")
    r
    "DOWN"))

(defn agg-status-check [status-obj]
  (let [values (vals status-obj)
        reduced-status (reduce reduce-helper "UP" values)]
    reduced-status))

(defn create-health-check [to-check]
  (fn [& args]
    (let [status-obj 
            (->> 
                {}
                (maybe-check-health to-check :db)
                (maybe-check-health to-check :rethink)
                (maybe-check-health to-check :rabbit)
                (maybe-check-health to-check :diskSpace)
                (maybe-check-health to-check :elasticSearch))
          total-status (agg-status-check status-obj)
          status-body (assoc status-obj :status total-status)]
    (if (= "UP" total-status)
      {
        :status 200
        :body  status-body }
      {
        :status 503
        :body  status-body }))))

(def health (create-health-check #{}))

(defn health-old [& args]
  {:status 200
   :body   {:status        "UP"
            :db            [{:database "mysql"
                             :status   (check-mysql-connection)}
                            {:database "rethink"
                             :status   (check-rethink-connection)}]
            :rabbit        (check-rabbit-connection)
            :diskSpace     (get-disk-space 0.90)
            :elasticSearch (get-elastic-search-connection)}})
