(ns utils.rabbit-service
  (:require [langohr.core :as rmq]
            [langohr.channel :as lch]
            [langohr.queue :as lq]
            [langohr.basic :as lb]
            [langohr.consumers :as lc]
            [langohr.exchange :as le]
            [cheshire.core :as json]
            [utils.logger :as logger]
            [utils.mdc :as mdc]
            [new-reliquary.core :refer [with-newrelic-transaction]])
  (:import [com.rabbitmq.client Channel Consumer DefaultConsumer QueueingConsumer
                                QueueingConsumer$Delivery ShutdownSignalException Envelope
                                AMQP$BasicProperties QueueingConsumer$Delivery]
           [java.util UUID]))

(def ^{:const true} default-exchange-name "")

(defn connection-settings [host-env username-env password-env]
  {:username (System/getenv username-env)
   :password (System/getenv password-env)
   :vhost    "/"
   :host     (System/getenv host-env)
   :port     5672})

(def ^:const RABBIT_TRANS "RabbitQ")

(def q-connection-setting (connection-settings "AMQPHOST" "AMQPUSER" "AMQPPASS"))

(defn make-mdc-headers []
  {:headers {"x-cs-request-id" (mdc/get-request-id) "x-cs-tenant-id" (mdc/get-tenant-id) "x-cs-user-id" (mdc/get-user-id)}})

(defn write-context-headers [headers]
  (when-let [x-cs-request-id (get headers "x-cs-request-id")]
    (mdc/put-request-id x-cs-request-id))
  (when-let [x-cs-user-id (get headers "x-cs-user-id")]
    (mdc/put-user-id x-cs-user-id))
  (when-let [x-cs-tenant-id (get headers "x-cs-tenant-id")]
    (mdc/put-tenant-id x-cs-tenant-id)))

(defn deliveries-seq
  "Builds a lazy seq of delivery instances from a queueing consumer."
  [^QueueingConsumer qcs timeoutms]
  (lazy-seq (cons (.nextDelivery qcs timeoutms) (deliveries-seq qcs timeoutms))))

(defn parse-rabbit-payload [payload]
  (if (nil? payload)
    nil
    (json/parse-string (String. payload "UTF-8") true)))

(defn message-topic-handler-wrapper [raw-handler ch {:keys [content-type delivery-tag type routing-key headers] :as meta} payload]
  (write-context-headers headers)
  (with-newrelic-transaction RABBIT_TRANS routing-key #(raw-handler (parse-rabbit-payload payload) routing-key)))

(defn message-handler-wrapper [raw-handler ch {:keys [content-type delivery-tag type routing-key headers] :as meta} payload]
  (write-context-headers headers)
  (with-newrelic-transaction RABBIT_TRANS routing-key #(raw-handler (parse-rabbit-payload payload))))

(defn message-rpc-handler-wrapper [raw-handler ch {:keys [content-type type delivery-tag reply-to correlation-id headers] :as meta} payload]
  (write-context-headers headers)
  (let [response (raw-handler (parse-rabbit-payload payload))]
    (lb/publish ch "" reply-to (json/generate-string response) (merge {:correlation-id correlation-id} (make-mdc-headers)))
    (lb/ack ch delivery-tag)))


(defn start-consumer
  [queue-name message-handler]
  (try
    (let [conn (rmq/connect (connection-settings "AMQPHOST" "AMQPUSER" "AMQPPASS"))
          ch (lch/open conn)
          callback (partial message-handler-wrapper message-handler)]
      (lq/declare ch queue-name {:exclusive false :auto-delete false :durable true})
      (lc/subscribe ch queue-name callback {:auto-ack true}))
    (catch Throwable th (logger/err th))))

;;TODO - Use exhanges
;;TODO - What if something dies on the queue...
(defn start-rpc-consumer
  [queue-name message-handler]
  (try
    (let [conn (rmq/connect (connection-settings "AMQPHOST" "AMQPUSER" "AMQPPASS"))
          ch (lch/open conn)
          callback (partial message-rpc-handler-wrapper message-handler)]
      (lq/declare ch queue-name {:auto-delete false})
      (lb/qos ch 1)
      (lc/blocking-subscribe ch queue-name callback))
    (catch Throwable th (logger/err th))))

(defn publish
  [message queue-name]
  (try
    (let [conn (rmq/connect (connection-settings "AMQPHOST" "AMQPUSER" "AMQPPASS"))
          ch (lch/open conn)]
      (lq/declare ch queue-name {:auto-delete false :durable true :exclusive false})
      (lb/publish ch default-exchange-name queue-name (json/generate-string message) (make-mdc-headers))
      (Thread/sleep 100)
      (rmq/close ch)
      (rmq/close conn))
    (catch Throwable th (logger/err th))))

(defn correlation-id-equals?
  [correlation-id d]
  (= (.getCorrelationId (.getProperties d)) correlation-id))

(defn nil-or-correlation-id-equals?
  [correlation-id d]
  (or (nil? d)
      (= (.getCorrelationId (.getProperties d)) correlation-id)))

(defn nil-or-body
  [d]
  (if (nil? d)
    nil
    (.getBody d)))

;;TODO - Setup like an exchange
;;TODO - How to setup a timeout
(defn publish-rpc
  [message queue-name timeoutms]
  (let [conn (rmq/connect (connection-settings "AMQPHOST" "AMQPUSER" "AMQPPASS"))
        ch (lch/open conn)
        cbq (lq/declare ch "" {:auto-delete false :exclusive true})
        consumer (lc/create-queueing ch {})]
    (let [correlation-id (str (UUID/randomUUID))]
      (lb/publish ch "" queue-name (json/generate-string message) (merge {:reply-to (:queue cbq) :correlation-id correlation-id} (make-mdc-headers)))
      (lb/consume ch (:queue cbq) consumer)
      (-> (->> (deliveries-seq consumer timeoutms)
               (filter (partial nil-or-correlation-id-equals? correlation-id))
               (first))
          nil-or-body
          (parse-rabbit-payload)))))

(defn publish->exchange
  ([type exchange-name routing-key message]
   (publish->exchange type exchange-name routing-key message {:content-type "application/json"}))
  ([type exchange-name routing-key message headers]
   (publish->exchange type exchange-name routing-key message headers {}))
  ([type exchange-name routing-key message headers x-options]
   (try
     (let [conn (rmq/connect (connection-settings "AMQPHOST" "AMQPUSER" "AMQPPASS"))
           ch (lch/open conn)]
       (le/declare ch exchange-name type (merge {:auto-delete false :durable true :exclusive false} x-options))
       (lb/publish ch exchange-name routing-key (json/generate-string message) (merge headers (make-mdc-headers)))
       (Thread/sleep 100)
       (rmq/close ch)
       (rmq/close conn))
     (catch Throwable th (logger/err th)))))

(defn bind->exchange
  [& {:keys [x-name x-type x-options queue-name queue-options topics sub-options handler] :or {queue-name "" topics [""]}}]
  (try
    (let [conn (rmq/connect q-connection-setting)
          ch (lch/open conn)
          queue-name' (:queue (lq/declare ch queue-name (merge {:auto-delete false :durable true :exclusive false} queue-options)))
          callback (partial message-topic-handler-wrapper handler)]
      (le/declare ch x-name x-type (merge {:auto-delete false :durable true :exclusive false} x-options))
      (doseq [topic topics]
        (lq/bind ch queue-name' x-name {:routing-key topic}))
      (lc/subscribe ch queue-name' callback (merge {} sub-options)))
    (catch Throwable th (logger/err th))))