(ns utils.mysql-service
  (:require [clojure.java.jdbc :as sql]
            [utils.logger :as logger]
            [camel-snake-kebab.core :as c]
            [camel-snake-kebab.extras :refer [transform-keys]])
  (:import (com.mchange.v2.c3p0 ComboPooledDataSource)))

(defn connection-config [host user pass]
  {:classname "com.mysql.jdbc.Driver"
   :subprotocol "mysql"
   :subname (str "//" (System/getenv host) ":3306/clientsuccess")
   :user (System/getenv user)
   :password (System/getenv pass)})

(defn pool [connection-info]
  {:datasource (doto
                 (ComboPooledDataSource.)
                 (.setDriverClass (:classname connection-info))
                 (.setJdbcUrl (str "jdbc:" (:subprotocol connection-info) ":" (:subname connection-info)))
                 (.setUser (:user connection-info))
                 (.setPassword (:password connection-info))
                 ;; expire excess connections after 30 minutes of inactivity:
                 (.setMaxIdleTimeExcessConnections (* 30 60))
                 ;; expire connections after 3 hours of inactivity:
                 (.setMaxIdleTime (* 3 60 60)))})

(def pooled-db-write (delay (pool (connection-config "DBHOST" "DBUSER" "DBPASS"))))
(def pooled-db-read (delay (pool (connection-config "DBRRHOST" "DBRRUSER" "DBRRPASS"))))

(defn conn-write [] @pooled-db-write)
(defn conn-read [] @pooled-db-read)

(defn run-query [query params] (transform-keys c/->kebab-case (sql/query (conn-read) (into [query] params))))

(defn run-query-master [query params] (sql/query (conn-write) (into [query] params))) ;scott will slap you if you use this

(defn insert-query [table-name insert-object]
  (if (seq? insert-object)
    (apply sql/insert! (conn-write) table-name insert-object)
    (sql/insert! (conn-write) table-name insert-object)))

(defn update-query [table-name update-object where-clause]
  (sql/update! (conn-write) table-name update-object where-clause))

(defn delete-query [table-name where-clause] (sql/delete! (conn-write) table-name where-clause))

(defn run-command! [query params] (sql/execute! (conn-write) (into [query] params)))
