(ns utils.mysql-service
  (:require [clojure.java.jdbc :as sql]
            [utils.logger :as logger])
  (:import (com.mchange.v2.c3p0 ComboPooledDataSource)))

(defn connection-config []
  {:classname "com.mysql.jdbc.Driver"
   :subprotocol "mysql"
   :subname (str "//" (System/getenv "DBRRHOST") ":3306/clientsuccess")
   :user (System/getenv "DBRRUSER")
   :password (System/getenv "DBRRPASS")})

(defn pool [connection-info]
  {:datasource (doto
                 (ComboPooledDataSource.)
                 (.setDriverClass (:classname connection-info))
                 (.setJdbcUrl (str "jdbc:" (:subprotocol connection-info) ":" (:subname connection-info)))
                 (.setUser (:user connection-info))
                 (.setPassword (:password connection-info))
                 ;; expire excess connections after 30 minutes of inactivity:
                 (.setMaxIdleTimeExcessConnections (* 30 60))
                 ;; expire connections after 3 hours of inactivity:
                 (.setMaxIdleTime (* 3 60 60)))})

(def pooled-db (delay (pool (connection-config))))

(defn conn [] @pooled-db)

(defn run-query [query params] (sql/query (conn) (into [query] params)))

(defn insert-query [table-name insert-object]
  (if (seq? insert-object)
    (apply sql/insert! (conn) table-name insert-object)
    (sql/insert! (conn) table-name insert-object)))

(defn update-query [table-name update-object where-clause]
  (sql/update! (conn) table-name update-object where-clause))

(defn delete-query [table-name where-clause] (sql/delete! (conn) table-name where-clause))
