(ns pedestal-utils.auth
  (:require [utils.employee-info-service :as employee]
            [io.pedestal.interceptor.chain :refer [terminate enqueue*]])
  (:import (org.slf4j MDC)))

(def ^:private tenant-id-field "tenantId")

(def ^:private employee-id-field "employeeId")

(def ^:private request-id-field "requestId")

(def remove-request-logging-fields []
  (MDC/remove tenant-id-field) (MDC/remove employee-id-field) (MDC/remove request-id-field))

(defn get-token [headers]
  (let [auth-header (get headers "authorization")
        bearer-length (count "Bearer ")]
    (if (not (nil? auth-header))
      (if (< bearer-length (count auth-header))
        (subs auth-header bearer-length)
        "")
      "")))

(def pedestal-401-response
  {:name  :401-response
   :leave (fn [context]
            (assoc context :response {:status  401
                                      :headers {}
                                      :body    {:userMessage "Unauthorized"}}))})

(def authenticate
  {:name  :auth-interceptor
   :enter (fn [{{headers :headers} :request :as context}]
            (let [token (get-token headers)
                  {{employee-id :id
                    tenant-id   :tenantId :as employee-info} :body
                   status                                    :status} (employee/do-remote-employee-info-call token)]
              (if (= status 200)
                (do (remove-request-logging-fields)
                    (MDC/put tenant-id-field tenant-id) (MDC/put request-id-field (:X-Request-Id headers)) (MDC/put employee-id-field employee-id)
                    (assoc-in context [:request :employee-info] employee-info))
                (-> context
                    terminate
                    (enqueue* pedestal-401-response)))))
   :leave (fn [context]
            (remove-request-logging-fields)
            context)})