(ns clanhr.reports-api.models.timelog
  "Timelog representation and validation"
  (:require [validateur.validation :refer :all]
            [clanhr.validators.core :as validators]
            [clj-time.coerce :as c]
            [clj-time.format :as f]
            [result.core :as result]))

(def timelog-validator (validation-set
                         (presence-of :account-id)
                         (presence-of :user-id)
                         (length-of :name :within (range 3 100))
                         (presence-of :start-time)
                         (validators/date-validator :start-time)
                         (presence-of :end-time)
                         (validators/date-validator :end-time)
                         (numericality-of :hours :only-integer false :gt 0)
                         ))

(defn- build-date
  "Builds a proper date"
  [timelog field]
  (let [raw (f/unparse (f/formatter "yyyy-MM-dd HH:mm:ss.1ZZ")
             (c/from-string (field timelog)))]
    raw))

(defn build
  "Builds a timelog with proper date objects"
  [timelog]
  (cond-> timelog
    (string? (:start-time timelog)) (assoc :start-time (build-date timelog :start-time))
    (string? (:end-time timelog)) (assoc :end-time (build-date timelog :end-time))))

(defn validate
  "Verifies if the absence's format is valid"
  [absence]
  (let [validation (timelog-validator absence)]
    (result/enforce-empty validation)))
