(ns clanhr.reports-api.models.expense
  "Expense representation and validation"
  (:require [validateur.validation :refer :all]
            [clanhr.validators.core :as validators]
            [clj-time.coerce :as c]
            [clj-time.format :as f]
            [result.core :as result]
            [schema.core :as s]))

(def schema
  "Representation of this model"
  {:id (s/either s/Uuid s/Str)
   :account-id (s/either s/Uuid s/Str)
   :user-id (s/either s/Uuid s/Str)
   :state s/Str
   :expense-date s/Any
   (s/optional-key :document-number) (s/maybe s/Str)
   :type s/Str
   :value s/Str
   :updated-at s/Any
   :created-at s/Any
   (s/optional-key :description) (s/maybe s/Str)})

(def expenses-validator (validation-set
                          (presence-of :id)
                          (presence-of :account-id)
                          (presence-of :user-id)
                          (presence-of :state)
                          (presence-of :expense-date)
                          (presence-of :type)
                          (presence-of :value)))

(defn- build-date
  "Builds a proper date"
  [timelog field]
  (let [raw (f/unparse (f/formatter "yyyy-MM-dd HH:mm:ss.1ZZ")
             (c/from-string (field timelog)))]
    raw))

(defn build
  "Builds a timelog with proper date objects"
  [expense]
  (cond-> expense
    (string? (:date-expense expense)) (assoc :date-expense (build-date expense :expense-date))))

(defn validate
  "Verifies if the expense format is valid"
  [model]
  (let [validation (expenses-validator model)]
    (result/enforce-empty validation)))
