(ns clanhr.reports-api.middlewares.context
  (:require [clanhr.reply.core :as reply]
            [clanhr.reply.json :as json]
            [result.core :as result]
            [clojure.core.async :refer [go <!]]
            [clanhr.analytics.errors :as errors]
            [clanhr.postgres-gateway.component :as pg-component]
            [clanhr.postgres-gateway.config :as pg-config]
            [clanhr.postgres-gateway.connection-provider :as connection-provider]
            [clanhr.reports-api.gateways.postgres.vacation-balance :as vacation-balance-postgres]
            [clanhr.reports-api.gateways.postgres.absence-report :as absence-report-postgres]
            [clanhr.reports-api.gateways.postgres.user-report :as user-postgres]
            [clanhr.reports-api.gateways.postgres.user-changes-report :as user-changes-postgres]
            [clanhr.reports-api.gateways.postgres.documents :as documents-postgres]
            [clanhr.reports-api.gateways.postgres.expenses :as expenses-postgres]
            [clanhr.reports-api.gateways.postgres.kms :as kms-postgres]
            [clanhr.reports-api.gateways.postgres.timelog-report :as timelog-postgres]
            [clanhr.reports-api.gateways.postgres.common :as postgres-common]
            [com.stuartsierra.component :as component]))

(defn extract-principal
  [context request]
  (assoc context :account-id (get-in request [:principal :account])
                 :user-id (get-in request [:principal :user-id])))

(defn build-context
  [request system]
  (-> system
      (extract-principal request)
      (assoc :token (or (:token request)
                        (get-in request [:headers "x-clanhr-auth-token"])
                        (get-in request [:query-params "token"])))
      (assoc :request request)))

(defn run
  "builds a context"
  [handler system]
  (fn [request]
    (let [context (build-context request system)]
      (handler (assoc request :context context)))))

(defn pg-system
  "Creates a pg-system"
  ([]
   (pg-system {}))
  ([args]
   (component/system-map
     :pg-conn (pg-component/create (postgres-common/config) {:async-close? args})
     :pg-stream-config (postgres-common/stream-config)
     :absence-report-gateway (absence-report-postgres/create)
     :user-report-gateway (user-postgres/create)
     :user-changes-report-gateway (user-changes-postgres/create)
     :kms-gateway (kms-postgres/create)
     :timelog-gateway (timelog-postgres/create)
     :documents-gateway (documents-postgres/create)
     :expenses-gateway (expenses-postgres/create)
     :vacations-balance-gateway (vacation-balance-postgres/create))))

(defn system
  "Builds the global context"
  []
  (pg-system))

(defn transaction-run!
  "Runs the given fn in a transaction. Is expected that fn returns a channel
  with a result"
  [context f]
  (pg-config/transaction-run! context f))
