(ns clanhr.reports-api.lib.utils
  "Utility fns repo"
  (:require [clojure.core.async :refer [go <!]]
            [clojure.string :as string]
            [clanhr.postgres-gateway.utils :as pg-utils]
            [result.core :as result]
            [clanhr.i18n.core :as i18n]
            [result.core :as result]
            [clj-time.coerce :as c]
            [clj-time.format :as f]))

(def query-defaults {:per-page 10
                     :page 1})

(defn uniformize-input-collection
  "Gets a uniformized version of the arg"
  [m k]
  (let [v (get m k)]
    (cond
      (seq v) (if (= "" (first v))
                []
                v)
      :else [])))

(defn calculate-pages
  "Calculates the number of possible pages"
  [total per-page]
  (let [total (or total 0)]
    (quot (- (+ total per-page) 1) per-page)))

(defn str->array
  "Transforms a string in an array"
  [raw]
  (if (string? raw)
    (clojure.string/split raw #",")
    raw))

(defn prepare-changed-value
  "Some changed values need special handling to be converted to strings"
  [raw]
  (cond
    (vector? raw) (string/join ", " raw)
    :else raw))

(defn like-str-switch
  "Helper that prepares a string argument and a switch for sql queries"
  [m k]
  (let [v (get m k)]
    [(pg-utils/like-value v) (if (empty? v) 1 0)]))

(defn str-switch
  "Helper that prepares a string argument and a switch for sql queries"
  [m k]
  (let [v (get m k)]
    [v (if (empty? v) 1 0)]))

(defn val-switch
  "Helper that prepares a generic value argument and a switch for sql queries"
  [m k]
  (let [v (get m k)]
    [v (if (nil? v) 1 0)]))

(def default-empty-uuid
  "Because we don't have a way to specify an empty uuid, this will act
  as the empty one"
  "2bcfb25e-22cc-41ca-a048-6e7d39bc41c4")

(defn uuid-switch
  "Helper that prepares an uuid argument and a switch for sql queries"
  [m k]
  (let [v (get m k default-empty-uuid)]
    [v (if (= v default-empty-uuid) 1 0)]))

(defn start-end-date-switch
  "Helper that prepares a start and end date, with switch"
  [m]
  (let [start-date (or (:start-date m) (:end-date m))
        end-date (or (:end-date m) (:start-date m))]
    [start-date
     end-date
     (if (and start-date end-date) 0 1)]))

(defn coll-switch
  "Helper that prepares a coll argument and a switch for sql queries"
  [m k]
  (let [raw (uniformize-input-collection m k)]
    [(pg-utils/array-column-value raw)
     (if (empty? raw) 1 0)]))

(defn get-page-info
  "Gathers page and per-page from given m. Returns numbers"
  [m]
  (let [page (:page m)
        per-page (:per-page m)]
    [(if (string? page) (Integer/parseInt page) page)
     (if (string? per-page) (Integer/parseInt per-page) per-page)]))

(defn paginated-query-result
  "Given the results of a paginated query and the total count, returns
  a proper result"
  [results total-results page per-page]
  (cond
    (result/failed? total-results) total-results
    (result/failed? results) results
    :else (result/success {:data (:data results)
                           :total-results (:data total-results)
                           :number-of-pages (calculate-pages (:data total-results) per-page)
                           :current-page page})))

(def custom-formatter (f/formatter "dd/MM/yyyy"))

(defn format-date
  [date]
  (->> date
       (c/from-string)
       (f/unparse custom-formatter)))

(defn format-interval
  [start-date end-date]
  (if (and (nil? start-date) (nil? end-date))
    ""
    (str (format-date start-date) " - " (format-date end-date))))

(defn get-name
  [lang key-name]
  (str (i18n/t lang key-name) ":"))

(def select-values (comp vals select-keys))

(defn format-address
  "Format the address field"
  [address]
  (let [street (:street address)
        parish (:parish address)
        district (:district address)
        zip-code (:zip-code address)
        city (:city address)]
    (string/join ", " (remove nil? [street parish district zip-code city]))))
