(ns clanhr.reports-api.gateways.postgres.vacation-balance
  "Postgres vacation gateway and component"
  (:require [clanhr.reports-api.gateways.vacation-balance :as vb]
            [clanhr.reports-api.lib.utils :as utils]
            [clanhr.postgres-gateway.core :as gateway]
            [clanhr.postgres-gateway.utils :as pg-utils]
            [clanhr.reports-api.gateways.postgres.common :as common]
            [com.stuartsierra.component :as component]
            [clj-time.coerce :as c]
            [clj-time.core :as t]
            [clojure.core.async :refer [go <! <!!]]
            [result.core :as result]))

(def absence-table-name common/vacations-balance-table-name)
(def user-table-name common/vacations-balance-user-table-name)

(defn date-year
  "Gets the date's year"
  [date]
  (t/year (c/from-date (c/to-date date))))

(defn- get-year-settings
  "Having an absences user, get which years have days configurated. If no
  days are present, then get some nice defaults"
  [user]
  (if-let [year-settings (:total-possible-days user)]
    year-settings
    {:default (or (:total-vacations user) (:total-days user))}))

(defn- parse-config-year
  "Given :default or :2015 will return the proper year to persist on the DB"
  [raw-year]
  (if (= :default raw-year)
    nil
    (name raw-year)))

(defn- save-user-year-vacations!
  "Save specific user vacations on a given year"
  [context user year value]
  (let [user-id (:_id user)
        year (parse-config-year year)]
    (gateway/save-data! {:id user-id
                         :name (:name user)
                         :account_id (:account-id user)
                         :total_days value
                         :teams (pg-utils/array-column-value (get-in user [:teams]))
                         :projects (pg-utils/array-column-value (get-in user [:projects]))
                         :tags (pg-utils/array-column-value (get-in user [:tags]))
                         :year year}
                        (assoc (common/pg-config-for user-table-name context)
                               :pk-query ["id = $1 and year = $2" user-id year]))))

(defn- save-user-vacations!
  "Setups total available vacation days per year for a given absences user"
  [context user]
  (let [years (get-year-settings user)
        persisted (doall (for [[year value] years]
                           (<!! (save-user-year-vacations! context user year value))))]
    (go (result/success persisted))))

(defn- delete-user!
  "Deletes a user from the report"
  [context user-id]
  (go
    (let [user-part (<! (gateway/update-models [(str "delete from " user-table-name " where id = $1") user-id] (common/pg-config-for user-table-name context)))
          absence-part (<! (gateway/update-models [(str "delete from " absence-table-name " where user_id = $1") user-id] (common/pg-config-for user-table-name context)))]
      (result/success))))

(defn- save-user-part!
  "Saves the absence part of the request"
  [context user absence]
  (let [user-id (:user-id absence)
        year (date-year (:start-date absence))]
    (gateway/save-data! {:id user-id
                         :name (:username absence)
                         :colaborator_id (get-in user [:company-data :colaborator-id])
                         :account_id (:account-id absence)
                         :total_days (:total-days absence)
                         :teams (pg-utils/array-column-value (get-in user [:company-data :teams]))
                         :projects (pg-utils/array-column-value (get-in user [:company-data :projects]))
                         :tags (pg-utils/array-column-value (get-in user [:company-data :tags]))
                         :year year}
                        (assoc (common/pg-config-for user-table-name context)
                               :pk-query ["id = $1 and year = $2" user-id year]))))

(defn- save-absence-part!
  "Saves the user part of the request"
  [context absence]
  (let [year (date-year (:start-date absence))]
    (gateway/save-data! {:id (:absence-id absence)
                         :user_id (:user-id absence)
                         :year year
                         :start_date (:start-date absence)
                         :duration_days (:duration absence)}
                        (common/pg-config-for absence-table-name context))))

(defn- get-raw-absence
  "Gets a specific absence part, by id"
  [context absence-id]
  (go
    (let [sql (str "select * from " absence-table-name " where id = $1")
          config (common/pg-config-for absence-table-name context)]
      (result/on-success[result (<! (gateway/query-data [sql absence-id] config))]
        (result/presence (first (:data result)))))))

(defn- get-raw-user
  "Gets a specific user part, by id"
  [context absence-result]
  (go
    (let [user-id (:user_id absence-result)
          sql (str "select * from " user-table-name " where id = $1")
          config (common/pg-config-for absence-table-name context)]
      (result/on-success[result (<! (gateway/query-data [sql user-id] config))]
        (result/success (first (:data result)))))))

(defrecord VacationBalancePostgresGateway [pg-conn]

  component/Lifecycle

  (start [this] this)
  (stop [this] this)

  vb/VacationBalance

  (update-user [this context user]
    (gateway/update-models [(str "update " user-table-name
                                 " set name=$1, teams=$2, projects=$3, tags=$4, colaborator_id=$5"
                                 " where id = $6")
                            (get-in user [:personal-data :name])
                            (pg-utils/array-column-value (get-in user [:company-data :teams]))
                            (pg-utils/array-column-value (get-in user [:company-data :projects]))
                            (pg-utils/array-column-value (get-in user [:company-data :tags]))
                            (get-in user [:company-data :colaborator-id])
                            (get-in user [:_id])]
                           (common/pg-config-for user-table-name context)))

  (setup-vacations [this context user]
    (if (:vacations-disabled user)
      (delete-user! context (:_id user))
      (save-user-vacations! context user)))

  (delete-user [this context user-id]
    (delete-user! context user-id))

  (add-absence [this context user absence]
    (go
      (result/enforce-let [absence-part (<! (save-absence-part! context absence))
                           user-part (<! (save-user-part! context user absence))]
        (result/success {:user-part user-part
                         :absence-part absence-part}))))

  (get-absence [this context absence-id]
    (go
      (result/enforce-let [absence-result (<! (get-raw-absence context absence-id))
                           user-result (<! (get-raw-user context absence-result))]
        (result/success {:absence-part absence-result
                         :user-part user-result}))))

  (remove-absence [this context absence-id]
    (go
      (let [sql (str "delete from " absence-table-name " where id = $1")
            config (common/pg-config-for absence-table-name context)
            result (<! (gateway/delete-models [sql absence-id] config))]
        result)))

  (query [this context args]
    (go
      (let [[page per-page] (utils/get-page-info args)
            account-id (:account-id args)

            date (or (:date args) (t/now))
            year (or (:year args) (date-year date))

            [raw-name name-switch] (utils/like-str-switch args :name)
            [teams teams-switch] (utils/coll-switch args :teams)
            [projects projects-switch] (utils/coll-switch args :projects)
            [tags tags-switch] (utils/coll-switch args :tags)

            config (common/pg-config-for absence-table-name context)

            results (gateway/query-data
                      [(str "select user_part.name,"
                                   "user_part.colaborator_id,"
                                   "user_part.total_days,"
                                   "sum(absence_part.enjoyed_days) as enjoyed_days,"
                                   "sum(absence_part.scheduled_days) as scheduled_days "
                            "from " user-table-name " as user_part "

                            ;; get the user entry for the given year; if there is
                            ;; no entry for that year, the NULL one should be
                            ;; choosen
                            "inner join ("
                              "select id, max(year) as year from " user-table-name " "
                               "where year = $3 or year is null "
                               "group by id) as user_filter "
                             "on user_part.id = user_filter.id and (user_part.year is not distinct from user_filter.year) "

                             ;; join with the absences to sum the enjoyed days
                             ;; and the scheduled days; the left join is to
                             ;; keep the user even without absences
                             "left join (select user_id,"
                                                "case when start_date<=DATE($1) then duration_days else 0 end as enjoyed_days,"
                                                "case when start_date>DATE($1) then duration_days else 0 end as scheduled_days "
                                          "from " absence-table-name
                                          " where year = $3) as absence_part "
                             "on user_part.id = absence_part.user_id "

                            " where "
                                   "account_id = $2 and "
                                   "(teams && $4 or 1=$5) and "
                                   "(projects && $6 or 1=$7) and "
                                   "(projects && $10 or teams && $10 or tags && $10 or 1=$11) and "
                                   "(name ilike $8 or 1=$9)"

                            "group by user_part.id, user_part.name, user_part.colaborator_id, user_part.total_days "
                            "order by user_part.name") date account-id year teams teams-switch projects projects-switch raw-name name-switch tags tags-switch]
                        (merge config {:page page :per-page per-page}))

            total-results (gateway/count-models
                            [(str "select count(distinct id) from " user-table-name
                                   ;" left join (select user_id from reports.vacations_balances where year = $2) as absence_part"
                                   ;" on id = absence_part.user_id"
                                  " where account_id = $1 and "
                                  "year = $2 and "
                                  "(teams && $3 or 1=$4) and "
                                  "(projects && $5 or 1=$6) and "
                                  "(projects && $9 or teams && $9 or tags && $9 or 1=$10) and "
                                  "(name ilike $7 or 1=$8)") account-id year teams teams-switch projects projects-switch raw-name name-switch tags tags-switch]
                            config)]

        (utils/paginated-query-result (<! results) (<! total-results) page per-page)))))

(defn create
  "Creates the gateway instance component"
  []
  (component/using (map->VacationBalancePostgresGateway {})
                   [:pg-conn]))
