(ns clanhr.reports-api.gateways.postgres.user-report
  "Postgres user report gateway and component"
  (:require [clanhr.reports-api.gateways.user-report :as ur]
            [clanhr.reports-api.gateways.postgres.common :as common]
            [clanhr.postgres-gateway.utils :as pg-utils]
            [clanhr.postgres-gateway.core :as gateway]
            [clanhr.reports-api.lib.utils :as utils]
            [com.stuartsierra.component :as component]
            [clojure.core.async :refer [go <!]]
            [result.core :as result]))

(def table-name common/user-report-table-name)

(defn update-user-config
  "Gets a config specific for persiting users"
  [table-name context user]
  (-> (common/pg-config-for table-name context)
      (assoc :fields {:name (get-in user [:personal-data :name])
                      :position (get-in user [:company-data :position])
                      :account_id (get-in user [:system :account])
                      :approver_id (get-in user [:company-data :approver-id])
                      :teams (pg-utils/array-column-value (get-in user [:company-data :teams]))
                      :tags (pg-utils/array-column-value (get-in user [:company-data :tags]))
                      :projects (pg-utils/array-column-value (get-in user [:company-data :projects]))})))

(defrecord UserPostgresGateway [pg-conn]

  component/Lifecycle

  (start [this] this)
  (stop [this] this)

  ur/UserReport

  (update-user [this context user]
      (gateway/save-model-with-id!
        user
        (update-user-config table-name context user)))

  (delete-user [this context user-id]
    (gateway/delete-models [(str "delete from " table-name " where id = $1")
                            user-id]
                           (common/pg-config-for table-name context)))

  (get-user [this context user-id]
    (gateway/query-one [(str "select model from " table-name " where id = $1") user-id] (common/pg-config-for table-name context)))

  (query [this context args]
    (go
      (let [account-id (:account-id args)

            [raw-name name-switch] (utils/like-str-switch args :name)
            [positions positions-switch] (utils/coll-switch args :positions)
            [managers-ids managers-ids-switch] (utils/coll-switch args :managers-ids)
            [teams teams-switch] (utils/coll-switch args :teams)
            [tags tags-switch] (utils/coll-switch args :tags)
            [projects projects-switch] (utils/coll-switch args :projects)

            [page per-page] (utils/get-page-info args)

            config (common/pg-config-for table-name context)

            results (gateway/query
                      [(str "select model from " table-name " "
                            "where account_id = $1 and "
                                  "(position = ANY($8) or 1=$9) and "
                                  "(approver_id = ANY($10) or 1=$11) and "
                                  "(teams && $12 or tags && $12 or projects && $12 or 1=$13) and "
                                  "(projects && $2 or 1=$3) and "
                                  "(teams && $4 or 1=$5) and "
                                  "(name ilike $6 or 1=$7)"
                            "order by name ASC")
                       account-id
                       projects projects-switch
                       teams teams-switch
                       raw-name name-switch
                       positions positions-switch
                       managers-ids managers-ids-switch
                       tags tags-switch]
                      (merge config {:page page :per-page per-page}))

            total-results (gateway/count-models
                            [(str "select count(*) from " table-name
                                  " where account_id = $1 and "
                                         "(position = ANY($8) or 1=$9) and "
                                         "(approver_id = ANY($10) or 1=$11) and "
                                         "(teams && $12 or tags && $12 or projects && $12 or 1=$13) and "
                                         "(projects && $2 or 1=$3) and "
                                         "(teams && $4 or 1=$5) and "
                                         "(name ilike $6 or 1=$7)"
                                         )
                             account-id
                             projects projects-switch
                             teams teams-switch
                             raw-name name-switch
                             positions positions-switch
                             managers-ids managers-ids-switch
                             tags tags-switch]
                            config)]

        (utils/paginated-query-result (<! results) (<! total-results) page per-page)))))

(defn create
  "Creates the gateway instance component"
  []
  (component/using (map->UserPostgresGateway {})
                   [:pg-conn]))
