(ns clanhr.reports-api.gateways.postgres.user-changes-report
  "Postgres user changes report gateway and component"
  (:require [clanhr.reports-api.gateways.user-report :as ur]
            [clanhr.reports-api.gateways.postgres.common :as common]
            [clanhr.postgres-gateway.utils :as pg-utils]
            [clanhr.postgres-gateway.core :as gateway]
            [clanhr.reports-api.lib.utils :as utils]
            [com.stuartsierra.component :as component]
            [clojure.core.async :refer [go <!]]
            [result.core :as result]))

(def table-name common/user-changes-report-table-name)

(defn- save-section-changes!
  "Gets the changes on a specified section and creates records for them"
  [context user changes section]
  (go
    (doseq [change (get changes section [])]
      (<! (gateway/save-data! {:id (java.util.UUID/randomUUID)
                               :name (get-in user [:personal-data :name])
                               :section (name section)
                               :field (:field change)
                               :old_value (utils/prepare-changed-value (:old change))
                               :new_value (utils/prepare-changed-value (:new change))
                               :user_id (get-in user [:_id])
                               :colaborator_id (get-in user [:company-data :colaborator-id])
                               :changer_id (or (get-in changes [:changer-id]) (get-in user [:_id]))
                               :position (get-in user [:company-data :position])
                               :account_id (get-in user [:system :account])
                               :approver_id (get-in user [:company-data :approver-id])
                               :teams (pg-utils/array-column-value (get-in user [:company-data :teams]))
                               :projects (pg-utils/array-column-value (get-in user [:company-data :projects]))
                               :tags (pg-utils/array-column-value (get-in user [:company-data :tags]))}
                              (common/pg-config-for table-name context))))))

(defn- update-user-changes-filters
  "Updates the changes already registerd on the database with the given
  users new updated fields"
  [context user]
  (gateway/update-models [(str "update " table-name "  "
                                     "set name=$1, teams=$2, tags=$3, colaborator_id=$4, "
                                          "position=$5, approver_id=$6, projects=$7 "
                                      "where user_id=$8")
                          (get-in user [:personal-data :name])
                          (pg-utils/array-column-value (get-in user [:company-data :teams]))
                          (pg-utils/array-column-value (get-in user [:company-data :tags]))
                          (get-in user [:company-data :colaborator-id])
                          (get-in user [:company-data :position])
                          (get-in user [:company-data :approver-id])
                          (pg-utils/array-column-value (get-in user [:company-data :projects]))
                          (get-in user [:_id])]
                         (common/pg-config-for table-name context)))

(defn- save-changes!
  "Gets the changes collections and creates records of each change"
  [context user changes]
  (go
    (<! (update-user-changes-filters context user))
    (<! (save-section-changes! context user changes :personal-data))
    (<! (save-section-changes! context user changes :company-data))
    (result/success :changes-saved)))

(defrecord UserChangesPostgresGateway [pg-conn]

  component/Lifecycle

  (start [this] this)
  (stop [this] this)

  ur/UserReport

  (update-user [this context user]
    (if-let [changes (:user-changes user)]
      (save-changes! context user changes)
      (go (result/success :nothing-to-do))))

  (delete-user [this context user-id]
    (gateway/delete-models [(str "delete from " table-name " where user_id = $1")
                            user-id]
                           (common/pg-config-for table-name context)))

  (get-user [this context user-id]
    (gateway/query [(str "select model from " table-name " where user_id = $1") user-id] (common/pg-config-for table-name context)))

  (query [this context args]
    (go
      (let [account-id (:account-id args)

            [raw-name name-switch] (utils/like-str-switch args :name)
            [positions positions-switch] (utils/coll-switch args :positions)
            [managers-ids managers-ids-switch] (utils/coll-switch args :managers-ids)
            [start-date end-date dates-switch] (utils/start-end-date-switch args)
            [teams teams-switch] (utils/coll-switch args :teams)
            [projects projects-switch] (utils/coll-switch args :projects)
            [tags tags-switch] (utils/coll-switch args :tags)

            [page per-page] (utils/get-page-info args)

            config (common/pg-config-for table-name context)

            jdbc-query [(str "select newer.colaborator_id, older.user_id, older.name, older.section, older.field, older.old_value, newer.new_value, older.position, older.teams, older.projects, older.tags, older.approver_id "
                            "from (select distinct on(user_id, section, field) user_id, section, field, old_value, name, created_at, account_id, position, teams, projects, tags,  approver_id "
                                   "from " table-name " "
                                   "order by user_id, section, field, created_at asc) as older "
                            "inner join (select distinct on(user_id, section, field) user_id, section, field, new_value, created_at, account_id, colaborator_id "
                                          "from " table-name " "
                                          "order by user_id, section, field, created_at desc) as newer "
                            "on newer.account_id = older.account_id and newer.user_id = older.user_id and newer.section = older.section and newer.field = older.field "
                            "where older.account_id = $1 and "
                                  "(older.position = ANY($8) or 1=$9) and "
                                  "(older.approver_id = ANY($10) or 1=$11) and "
                                  "(older.projects && $15 or older.teams && $15 or older.tags && $15 or 1=$16) and "
                                  "(older.projects && $2 or 1=$3) and "
                                  "(older.teams && $4 or 1=$5) and "
                                  "(older.name ilike $6 or 1=$7) and "
                                  "(older.old_value IS DISTINCT FROM newer.new_value) and "
                                  "(1=$14 or"
                                   "(DATE(newer.created_at) >= DATE($12) and DATE(newer.created_at) <= DATE($13))) "
                            "order by newer.created_at ASC, older.name ASC")
                       account-id
                       projects projects-switch
                       teams teams-switch
                       raw-name name-switch
                       positions positions-switch
                       managers-ids managers-ids-switch
                       start-date end-date dates-switch
                       tags tags-switch]

            results (gateway/query-data
                      jdbc-query
                      (merge config {:page page :per-page per-page}))

            total-results (gateway/count-models
                            (update jdbc-query 0 #(str "select count(*) from (" % ") as aux"))
                            config)]

        (utils/paginated-query-result (<! results) (<! total-results) page per-page)))))

(defn create
  "Creates the gateway instance component"
  []
  (component/using (map->UserChangesPostgresGateway {})
                   [:pg-conn]))
