(ns clanhr.reports-api.gateways.postgres.timelog-report
  "Postgres absence report gateway and component"
  (:require [clanhr.reports-api.gateways.timelog-report :as tr]
            [clanhr.reports-api.lib.utils :as utils]
            [clanhr.reports-api.gateways.postgres.common :as common]
            [clanhr.postgres-gateway.utils :as pg-utils]
            [clanhr.postgres-gateway.core :as gateway]
            [com.stuartsierra.component :as component]
            [clojure.core.async :refer [go <!]]
            [result.core :as result]))

(def table-name common/timelog-report-table-name)

(defrecord TimeLogPostgresGateway [pg-conn]

  component/Lifecycle

  (start [this] this)
  (stop [this] this)

  tr/TimeLogReport

  (update-user [this context user]
    (gateway/update-models [(str "update " table-name
                                 " set name=$1, teams=$2, tags=$3, projects=$4"
                                 " where user_id = $5")
                            (get-in user [:personal-data :name])
                            (pg-utils/array-column-value (get-in user [:company-data :teams]))
                            (pg-utils/array-column-value (get-in user [:company-data :projects]))
                            (pg-utils/array-column-value (get-in user [:company-data :tags]))
                            (get-in user [:_id])]
                           (common/pg-config-for table-name context)))

  (delete-user [this context user-id]
    (gateway/delete-models [(str "delete from " table-name " where user_id = $1")
                            user-id]
                           (common/pg-config-for table-name context)))

  (add-timelog [this context user timelog]
    (gateway/save-data! {:id (java.util.UUID/randomUUID)
                         :name (:name timelog)
                         :user_id (:user-id timelog)
                         :account_id (:account-id timelog)
                         :start_time (:start-time timelog)
                         :end_time (:end-time timelog)
                         :teams (pg-utils/array-column-value (get-in user [:company-data :teams]))
                         :projects (pg-utils/array-column-value (get-in user [:company-data :projects]))
                         :tags (pg-utils/array-column-value (get-in user [:company-data :tags]))
                         :hours (:hours timelog)}
                        (common/pg-config-for table-name context)))

  (get-timelog [this context timelog-id]
    (go
      (let [sql (str "select * from " table-name " where id = $1")
            config (common/pg-config-for table-name context)
            result (<! (gateway/query-data [sql timelog-id] config))
            timelog (first (:data result))]
        (if (and (result/succeeded? result) timelog)
          (result/success timelog)
          (result/failure "not-found")))))

  (query [this context args]
    (go
      (let [page (:page args)
            per-page (:per-page args)
            account-id (:account-id args)

            raw-name (pg-utils/like-value (:name args))
            name-switch (if (empty? (:name args)) 1 0)

            start-date (or (:start-date args) (:end-date args))
            end-date (or (:end-date args) (:start-date args))
            dates-switch (if (and start-date end-date) 0 1)

            raw-projects (utils/uniformize-input-collection args :projects)
            projects (pg-utils/array-column-value raw-projects)
            projects-switch (if (empty? raw-projects) 1 0)

            raw-teams (utils/uniformize-input-collection args :teams)
            teams (pg-utils/array-column-value raw-teams)
            teams-switch (if (empty? raw-teams) 1 0)

            raw-tags (utils/uniformize-input-collection args :tags)
            tags (pg-utils/array-column-value raw-tags)
            tags-switch (if (empty? raw-tags) 1 0)

            parsed-per-page (if (string? per-page) (Integer/parseInt per-page) per-page)
            parsed-page (if (string? page) (Integer/parseInt page) page)
            config (common/pg-config-for table-name context)

            results (gateway/query-data
                      [(str "select * from " table-name " "
                            "where account_id = $1 and "
                                  "(projects && $2 or teams && $2 or tags && $2 or 1=$3) and "
                                  "(projects && $4 or 1=$5) and "
                                  "(teams && $6 or 1=$7) and "
                                  "(name ilike $8 or 1=$9) and "
                                  "(1=$12 or "
                                   "(DATE($10) < DATE(start_time) and DATE($11) > DATE(end_time)) or "
                                   "(DATE($10) >= DATE(start_time) and DATE($10) <= DATE(end_time)) or "
                                   "(DATE($11) >= DATE(start_time) and DATE($11) <= DATE(end_time)))"
                            "order by name ASC, start_time ASC") account-id tags tags-switch projects projects-switch teams teams-switch raw-name name-switch start-date end-date dates-switch]
                      (merge config {:page page :per-page per-page}))

            total-results (gateway/count-models
                            [(str "select count(*) from " table-name
                                  " where account_id = $1 and "
                                          "(projects && $2 or teams && $2 or tags && $2 or 1=$3) and "
                                          "(projects && $4 or 1=$5) and "
                                          "(teams && $6 or 1=$7) and "
                                          "(name ilike $8 or 1=$9) and "
                                          "(1=$12 or "
                                           "(DATE($10) < DATE(start_time) and DATE($11) > DATE(end_time)) or "
                                           "(DATE($10) >= DATE(start_time) and DATE($10) <= DATE(end_time)) or "
                                           "(DATE($11) >= DATE(start_time) and DATE($11) <= DATE(end_time)))"
                                         ) account-id tags tags-switch projects projects-switch teams teams-switch raw-name name-switch start-date end-date dates-switch]
                            config)

            results (<! results)
            total-results (<! total-results)]

        (cond
          (result/failed? total-results) total-results
          (result/failed? results) results
          :else (result/success {:data (:data results)
                                 :total-results (:data total-results)
                                 :number-of-pages (utils/calculate-pages (:data total-results) parsed-per-page)
                                 :current-page parsed-page}))))))

(defn create
  "Creates the gateway instance component"
  []
  (component/using (map->TimeLogPostgresGateway {})
                   [:pg-conn]))
