(ns clanhr.reports-api.gateways.postgres.kms
  "Postgres kms gateway and component"
  (:require [clanhr.reports-api.gateways.kms :as kg]
            [clanhr.postgres-gateway.core :as gateway]
            [clanhr.reports-api.lib.utils :as utils]
            [clanhr.postgres-gateway.utils :as pg-utils]
            [clanhr.reports-api.gateways.postgres.common :as common]
            [com.stuartsierra.component :as component]
            [clj-time.coerce :as c]
            [clj-time.core :as t]
            [clojure.core.async :refer [go <! <!!]]
            [result.core :as result]))

(def table-name common/kms-report-table-name)

(defrecord KmsPostgresGateway [pg-conn]

  component/Lifecycle

  (start [this] this)
  (stop [this] this)

  kg/Kms

  (add-kms [this context kms]
    (gateway/save-data! {:id (:id kms)
                         :user_id (:user-id kms)
                         :account_id (:account-id kms)
                         :km_date (:date kms)
                         :license_plate (:license-plate kms)
                         :origin (:origin kms)
                         :destination (:destination kms)
                         :reason (:reason kms)
                         :state (:state kms)
                         :number_of_kms (:number-of-kms kms)
                         :value_per_km (:value-per-km kms)
                         :total_with_tax (:total-with-tax kms)}
                        (common/pg-config-for table-name context)))

  (query [this context args]
    (go
      (let [page (:page args)
            per-page (:per-page args)
            account-id (:account-id args)
            user-id (:user-id args)

            [km-status km-status-switch] (utils/coll-switch args :status)
            [km-start-date km-start-date-switch] (utils/val-switch args :start-date)
            [km-end-date km-end-date-switch] (utils/val-switch args :end-date)

            config (common/pg-config-for table-name context)

            results (gateway/query-data
                      [(str "select * from " table-name " "
                            "where account_id = $1 and user_id = $2 "
                            "and (state = ANY($3) or 1=$4) "
                            "and (km_date >= $5 or 1=$6) "
                            "and (km_date <= $7 or 1=$8) "
                            "order by km_date asc")
                        account-id user-id km-status km-status-switch
                        km-start-date km-start-date-switch
                        km-end-date km-end-date-switch]
                      (merge config {:page page :per-page per-page}))
            results (<! results)]

        (cond
          (result/failed? results) results
          :else (result/success {:data (map pg-utils/->lisp-case-keys (:data results))
                                 :total-results (count (:data results))
                                 :current-page 1}))))))

(defn create
  "Creates the gateway instance component"
  []
  (component/using (map->KmsPostgresGateway {})
                   [:pg-conn]))
