(ns clanhr.reports-api.gateways.postgres.expenses
  "Postgres expenses gateway and component"
  (:require [clanhr.reports-api.gateways.expenses :as eg]
            [clanhr.postgres-gateway.core :as gateway]
            [clanhr.postgres-gateway.utils :as pg-utils]
            [clanhr.reports-api.lib.utils :as utils]
            [clanhr.reports-api.gateways.postgres.common :as common]
            [com.stuartsierra.component :as component]
            [clj-time.coerce :as c]
            [clj-time.core :as t]
            [clojure.core.async :refer [go <! <!!]]
            [result.core :as result]))

(def table-name common/expenses-report-table-name)

(defrecord ExpensesPostgresGateway [pg-conn]

  component/Lifecycle

  (start [this] this)
  (stop [this] this)

  eg/Expenses

  (add-expense [this context expense]
    (gateway/save-data! {:id (or (:id expense) (java.util.UUID/randomUUID))
                         :user_id (:user-id expense)
                         :account_id (:account-id expense)
                         :state (:state expense)
                         :expense_date (:expense-date expense)
                         :type (:type expense)
                         :value (:value expense)
                         :document_number (:document-number expense)
                         :description (:description expense)}
                        (common/pg-config-for table-name context)))

  (query [this context args]
    (go
      (let [page (or (:page args) 1)
            per-page (or (:per-page args) 10)
            user-id (:user-id args)
            account-id (:account-id args)

            config (common/pg-config-for table-name context)

            [expense-status expense-status-switch] (utils/coll-switch args :status)
            [expense-types expense-types-switch] (utils/coll-switch args :types)
            [expense-start-date expense-start-date-switch] (utils/val-switch args :start-date)
            [expense-end-date expense-end-date-switch] (utils/val-switch args :end-date)

            results (gateway/query-data
                      [(str "select * from " table-name "  "
                            "where account_id = $1 and user_id = $2 "
                            "and (state = ANY($3) or 1=$4) and (type = ANY($5) or 1=$6) "
                            "and (expense_date >= $7 or 1=$8) "
                            "and (expense_date <= $9 or 1=$10) "
                            "order by expense_date asc")
                       account-id user-id expense-status expense-status-switch
                       expense-types expense-types-switch
                       expense-start-date expense-start-date-switch
                       expense-end-date expense-end-date-switch
                       ]
                      (merge config {:page page :per-page per-page}))

            results (<! results)]

        (cond
          (result/failed? results) results
          :else (result/success {:data (map pg-utils/->lisp-case-keys (:data results))
                                 :total-results (count (:data results))
                                 :current-page 1}))))))

(defn create
  "Creates the gateway instance component"
  []
  (component/using (map->ExpensesPostgresGateway {})
                   [:pg-conn]))
