(ns clanhr.reports-api.gateways.postgres.absence-report
  "Postgres absence report gateway and component"
  (:require [clanhr.reports-api.gateways.absence-report :as ar]
            [clanhr.reports-api.gateways.postgres.common :as common]
            [clanhr.postgres-gateway.utils :as pg-utils]
            [clanhr.postgres-gateway.core :as gateway]
            [clanhr.reports-api.lib.utils :as utils]
            [com.stuartsierra.component :as component]
            [clojure.core.async :refer [go <!]]
            [result.core :as result]))

(def table-name common/absence-report-table-name)

(defrecord AbsenceReportPostgresGateway [pg-conn]

  component/Lifecycle

  (start [this] this)
  (stop [this] this)

  ar/AbsenceReport

  (update-user [this context user]
    (gateway/update-models [(str "update " table-name
                                 " set name=$1, teams=$2, projects=$3, tags=$4, colaborator_id=$5, manager_ids=$7"
                                 " where user_id = $6")
                            (get-in user [:personal-data :name])
                            (pg-utils/array-column-value (get-in user [:company-data :teams]))
                            (pg-utils/array-column-value (get-in user [:company-data :projects]))
                            (pg-utils/array-column-value (get-in user [:company-data :tags]))
                            (get-in user [:company-data :colaborator-id])
                            (get-in user [:_id])
                            (pg-utils/array-column-value (map str (get-in user [:company-data :manager-ids])))]
                           (common/pg-config-for table-name context)))

  (delete-user [this context user-id]
    (gateway/delete-models [(str "delete from " table-name " where user_id = $1")
                            user-id]
                           (common/pg-config-for table-name context)))

  (add-absence [this context user absence]
    (gateway/save-data! {:id (:absence-id absence)
                         :name (:username absence)
                         :user_id (:user-id absence)
                         :colaborator_id (get-in user [:company-data :colaborator-id])
                         :account_id (:account-id absence)
                         :absence_type (:type absence)
                         :start_date (:start-date absence)
                         :teams (pg-utils/array-column-value (get-in user [:company-data :teams]))
                         :projects (pg-utils/array-column-value (get-in user [:company-data :projects]))
                         :tags (pg-utils/array-column-value (get-in user [:company-data :tags]))
                         :manager_ids (pg-utils/array-column-value (map str (get-in user [:company-data :manager-ids])))
                         :end_date (:end-date absence)
                         :duration (:duration absence)
                         :duration_type (:duration-type absence)}
                        (common/pg-config-for table-name context)))

  (get-absence [this context absence-id]
    (go
      (let [sql (str "select * from " table-name " where id = $1")
            config (common/pg-config-for table-name context)
            result (<! (gateway/query-data [sql absence-id] config))
            absence (first (:data result))]
        (if (and (result/succeeded? result) absence)
          (result/success absence)
          (result/failure "not-found")))))

  (remove-absence [this context absence-id]
    (go
      (let [sql (str "delete from " table-name " where id = $1")
            config (common/pg-config-for table-name context)
            result (<! (gateway/delete-models [sql absence-id] config))]
        result)))

  (query [this context args]
    (go
      (let [[page per-page] (utils/get-page-info args)
            account-id (:account-id args)

            [raw-name name-switch] (utils/like-str-switch args :name)
            [absence-type absence-type-switch] (utils/val-switch args :type)
            [absence-types absence-types-switch] (utils/coll-switch args :types)
            [start-date end-date dates-switch] (utils/start-end-date-switch args)
            [teams teams-switch] (utils/coll-switch args :teams)
            [projects projects-switch] (utils/coll-switch args :projects)
            [tags tags-switch] (utils/coll-switch args :tags)
            [manager-ids manager-ids-switch] (utils/coll-switch args :manager-ids)

            config (common/pg-config-for table-name context)

            results (gateway/query-data
                      [(str "select * from " table-name " "
                            "where account_id = $1 and "
                                  "(projects && $2 or teams && $2 or tags && $2 or 1=$3) and "
                                  "(projects && $4 or 1=$5) and "
                                  "(teams && $6 or 1=$7) and "
                                  "(manager_ids && $17 or user_id::varchar = ANY($17) or 1=$18) and "
                                  "(name ilike $8 or 1=$9) and "
                                  "(absence_type = $10 or 1=$11) and "
                                  "(absence_type = ANY($15) or 1=$16) and "
                                  "(1=$14 or"
                                   "(DATE($12) < start_date and DATE($13) > end_date) or "
                                   "(DATE($12) >= start_date and DATE($12) <= end_date) or "
                                   "(DATE($13) >= start_date and DATE($13) <= end_date))"
                            "order by name ASC, start_date ASC")
                       account-id
                       tags tags-switch
                       projects projects-switch
                       teams teams-switch
                       raw-name name-switch
                       absence-type absence-type-switch
                       start-date end-date dates-switch
                       absence-types absence-types-switch
                       manager-ids manager-ids-switch]
                      (merge config {:page page :per-page per-page}))

            total-results (gateway/count-models
                            [(str "select count(*) from " table-name
                                  " where account_id = $1 and "
                                          "(projects && $2 or teams && $2 or tags && $2 or 1=$3) and "
                                          "(projects && $4 or 1=$5) and "
                                          "(teams && $6 or 1=$7) and "
                                          "(manager_ids && $17 or user_id::varchar = ANY($17) or 1=$18) and "
                                          "(name ilike $8 or 1=$9) and "
                                          "(absence_type = $10 or 1=$11) and "
                                          "(absence_type = ANY($15) or 1=$16) and "
                                          "(1=$14 or"
                                           "(DATE($12) < start_date and DATE($13) > end_date) or "
                                           "(DATE($12) >= start_date and DATE($12) <= end_date) or "
                                           "(DATE($13) >= start_date and DATE($13) <= end_date))"
                                         )
                             account-id
                             tags tags-switch
                             projects projects-switch
                             teams teams-switch
                             raw-name name-switch
                             absence-type absence-type-switch
                             start-date end-date dates-switch
                             absence-types absence-types-switch
                             manager-ids manager-ids-switch]
                            config)]

        (utils/paginated-query-result (<! results) (<! total-results) page per-page)))))

(defn create
  "Creates the gateway instance component"
  []
  (component/using (map->AbsenceReportPostgresGateway {})
                   [:pg-conn]))
