(ns clanhr.reports-api.core.register-absence
  "Registers absences on the gateways that are related to absences"
  (:refer-clojure :exclude [run!])
  (:require [result.core :as result]
            [clojure.core.async :refer [go <!]]
            [clanhr-api.core :as clanhr-api]
            [clanhr.reports-api.models.absence :as absence-model]
            [clanhr.reports-api.middlewares.context :as context]
            [clanhr.reports-api.gateways.vacation-balance :as vbg]
            [clanhr.reports-api.gateways.absence-report :as arg]))

;; --
;; Vacations balance specific logic
;; --

(defn save-vacations-balance?
  "True if the absence is to be persisted on the vacations-balance"
  [context absence]
  (and (or (nil? (:duration absence))
                 (pos? (:duration absence)))
       (= "vacations" (:type absence))
       (= "approved" (:state absence))))

(defn remove-vacations-balance?
  "True if the absence is to be removed from the absence-report"
  [context absence]
  (or (= 0 (:duration absence))
      (= "pendent" (:state absence))
      (= "repproved" (:state absence))
      (= "canceled" (:state absence))))

(defn register-vacations-balance
  "Registers the absence as an vacations balance"
  [{:keys [vacations-balance-gateway] :as context} user absence]
  (cond
    (save-vacations-balance? context absence)
      (vbg/add-absence vacations-balance-gateway context user absence)
    (remove-vacations-balance? context absence)
      (vbg/remove-absence vacations-balance-gateway context (:absence-id absence))
    :else
      (go (result/success "vacations-balance-no-action"))))

;; --
;; Absence report specific logic
;; --

(defn save-absence-report?
  "True if the absence is to be persisted on the absence-report"
  [context absence]
  (and (= "approved" (:state absence))))

(defn remove-absence-report?
  "True if the absence is to be removed from the absence-report"
  [context absence]
  (remove-vacations-balance? context absence))

(defn register-absence-report
  "Registers the absence as an absence-report"
  [{:keys [absence-report-gateway] :as context} user absence]
  (cond
    (save-absence-report? context absence)
      (arg/add-absence absence-report-gateway context user absence)
    (remove-absence-report? context absence)
      (arg/remove-absence absence-report-gateway context (:absence-id absence))
    :else
      (go (result/success "absence-report-no-action"))))

;; --
;; Common logic
;; --

(defn validate
  "Validates the given data"
  [context absence]
  (absence-model/validate absence))

(defn- persist!
  "Processes the inserts"
  [context user absence]
  (go
    (result/enforce-let [absence-report (<! (register-absence-report context user absence))
                         vacations-balance (<! (register-vacations-balance context user absence))]
      (result/success {:absence-report absence-report
                       :vacations-balance vacations-balance}))))

(defn- get-user
  "Gets the user information for the given absence"
  [context absence]
  (if-let [result (:get-user-result context)]
    (go result)
    (clanhr-api/http-get {:service :directory-api
                          :path (str "/user/" (:user-id absence))
                          :token (:token context)})))

(defn build-absence
  "Prepares the given absence"
  [absence]
  (result/presence
    (if (nil? (:end-date absence))
      (assoc absence :end-date (:start-date absence))
      absence)))

(defn atomic-run!
  "Registers an absence."
  [context absence]
  (go
    (result/enforce-let [absence (build-absence absence)
                         valid? (validate context absence)
                         user (<! (get-user context absence))]
      (<! (context/transaction-run! context
                                    (fn [context] (persist! context user absence)))))))

