(ns clanhr.reports-api.core.list-vacations-balance
  "List vacations balance report"
  (:refer-clojure :exclude [run!])
  (:require [clojure.core.async :refer [go]]
            [result.core :as result]
            [clanhr.auth.authorized :as auth]
            [clanhr.reports-api.lib.metrics :as metrics]
            [clojure.core.async :refer [<! go]]
            [clanhr.reports-api.gateways.vacation-balance :as vbg]))

(def ^:private query-defaults {:per-page 10
                               :page 1})

(defn query-builder
  [context data]
  (let [query {:per-page (:per-page data)
               :page (:page data)
               :year (:year data)
               :tags (:tags data)
               :teams (:teams data)
               :name (:name data)
               :projects (:projects data)
               :account-id (:account-id context)}]
    (apply dissoc query
                  (for [[k v] query :when (nil? v)] k))))

(defn prepare-query
  "Ensure presence of per-page and page query params"
  [context data]
  (merge query-defaults (query-builder context data)))

(defn proper-value
  [k v]
  (if (= k :total_days)
    (if (== (int v) v)
      (int v)
      (float v))
    v))

(defn cleanup-row
  "Cleans up a row"
  [row]
  (into {}
    (map (fn [[k v]]
                [k (if (nil? v) 0 (proper-value k v))]) row)))

(defn- cleanup
  "Fixes and transforms data if necessary"
  [result]
  (let [data (:data result)]
    (assoc result :data (map cleanup-row data))))

(defn run!
  "List notifications in state to-process"
  [{:keys [vacations-balance-gateway] :as context} data]
  (go
    (result/enforce-let [result (<! (auth/authorized? (assoc context :action :reports-access)))]
      (metrics/report-generated "vacation-balance" data)
      (-> (<! (vbg/query vacations-balance-gateway context (prepare-query context data)))
          (cleanup)))))
