(ns clanhr.reports-api.core.list-vacations-and-absences
  "List vacations and absences report"
  (:refer-clojure :exclude [run!])
  (:require [clojure.core.async :refer [go]]
            [result.core :as result]
            [clj-time.coerce :as c]
            [clj-time.core :as t]
            [clanhr.analytics.errors :as errors]
            [clanhr.reports-api.lib.utils :as utils]
            [clanhr.reports-api.lib.metrics :as metrics]
            [clanhr.auth.authorized :as auth]
            [clanhr.work-days.core :as work-days]
            [clojure.core.async :refer [<! go]]
            [clanhr.reports-api.core.get-account-settings :as get-account-settings]
            [clanhr.reports-api.gateways.absence-report :as arg]))

(def ^:private query-defaults {:per-page 10
                               :page 1})

(defn query-builder
  [context data]
  (let [query {:per-page (:per-page data)
               :page (:page data)
               :name (:name data)
               :teams (:teams data)
               :projects (:projects data)
               :tags (:tags data)
               :manager-ids (:manager-ids data)
               :type (:type data)
               :types (utils/str->array (:types data))
               :start-date (c/from-string (:start-date data))
               :end-date (c/from-string (:end-date data))
               :account-id (:account-id context)}]
    (apply dissoc query
                  (for [[k v] query :when (nil? v)] k))))

(defn prepare-query
  "Ensure presence of per-page and page query params"
  [context data]
  (let [query (merge query-defaults (query-builder context data))]
    (cond
      (and (:start-date query)
           (:end-date query)
           (t/before? (:end-date query) (:start-date query)))
        (result/failure "start-date-after-end-date")

      :else
        (result/success query))))

(defn- absence-db->absence
  "Transforms the absence from the DB in a regular absence"
  [absence]
  (clojure.set/rename-keys absence {:duration_type :duration-type
                                    :absence_type :absence-type
                                    :start_date :start-date
                                    :end_date :end-date}))

(defn choose-older
  "Returns the older of the two dates"
  [query-date start-date]
  (if (or (nil? query-date) (t/after? start-date query-date))
    start-date
    query-date))

(defn choose-youngest
  "Returns the youngest of the two dates"
  [query-date end-date]
  (if (or (nil? query-date) (t/after? query-date end-date))
    end-date
    query-date))

(defn use-plain-duration?
  "True if we should use the duration and not try to calculate based on settings
  and dates."
  [absence]
  (boolean (some #{"hours" "partial-day"} [(:duration-type absence)])))

(defn absence-duration
  "Gets the absence duration"
  [settings absence start-date end-date]
  (if (use-plain-duration? absence)
    (:duration absence)
    (work-days/calculate settings (assoc absence :start-date start-date :end-date end-date))))

(defn- adjust-duration
  "Getst the work days duration"
  [query settings absence]
  (try
    (let [absence (-> (absence-db->absence absence)
                      (work-days/build))
          start-date (choose-older (:start-date query) (:start-date absence))
          end-date (choose-youngest (:end-date query) (:end-date absence))
          duration (absence-duration settings absence start-date end-date)]
      (assoc absence :duration duration
                     :start_date start-date
                     :end_date end-date))
    (catch Exception ex
      (throw (ex-info "Error adjusting absence duration" absence ex)))))

(defn calculate-work-days
  "Considers the start and end date and adjusts the duration for that period"
  [result query settings]
  (try
    (let [data (:data result)]
      (assoc result :data (mapv (partial adjust-duration query settings) data)))
    (catch Exception ex
      (errors/exception ex))))

(defn run!
  "List notifications in state to-process"
  [{:keys [absence-report-gateway] :as context} data]
  (go
    (result/enforce-let [query (prepare-query context data)
                         auth? (<! (auth/authorized? (assoc context :action :reports-access)))
                         settings (<! (get-account-settings/run context))
                         result (<! (arg/query absence-report-gateway context query))]
      (metrics/report-generated "absences" data)
      (calculate-work-days result query settings))))
