(ns clanhr.reports-api.core.list-user-changes
  "List user changed"
  (:refer-clojure :exclude [run!])
  (:require [clojure.core.async :refer [go]]
            [result.core :as result]
            [clj-time.coerce :as c]
            [clj-time.format :as f]
            [clj-time.core :as t]
            [clanhr.reports-api.lib.utils :as utils]
            [clanhr.reports-api.lib.metrics :as metrics]
            [clanhr.auth.authorized :as auth]
            [clanhr.i18n.core :as i18n]
            [clanhr.work-days.core :as work-days]
            [clojure.core.async :refer [<! go]]
            [clanhr.reports-api.core.get-account-settings :as get-account-settings]
            [clanhr.reports-api.gateways.user-report :as urg]))

(defn query-builder
  [context data]
  (let [query {:per-page (:per-page data)
               :page (:page data)
               :name (:name data)
               :positions (:positions data)
               :teams (:teams data)
               :projects (:projects data)
               :tags (:tags data)
               :start-date (:start-date data)
               :end-date (:end-date data)
               :managers-ids (:managers-ids data)
               :account-id (:account-id context)}]
    (apply dissoc query
                  (for [[k v] query :when (nil? v)] k))))

(defn prepare-query
  "Ensure presence of per-page and page query params"
  [context data]
  (result/success (merge utils/query-defaults (query-builder context data))))

(defn- convert-date
  "Converts a raw value to a date representaion"
  [raw]
  (->> raw
       (c/to-date)
       (c/from-date)
       (f/unparse (f/formatters :date))))

(defn- prepare-date
  "Prepares a date item"
  [item]
  (cond-> item
    (some? (:new_value item)) (assoc :new_value (convert-date (:new_value item)))
    (some? (:old_value item)) (assoc :old_value (convert-date (:old_value item)))))

(defn- prepare-translate
  "Prepares an item to be translated"
  [lang item]
  (cond-> item
    (some? (:new_value item)) (assoc :new_value (i18n/t lang (:new_value item)))
    (some? (:old_value item)) (assoc :old_value (i18n/t lang (:old_value item)))))

(defn- date-field?
  "True if a field is from a date"
  [field]
  (some #{(keyword field)} [:birth-date]))

(defn- translatable-field?
  "True if a field is to be translated"
  [field]
  (some #{(keyword field)} [:civil-status]))

(defn prepare-item
  "Prepares an item for output"
  [lang item]
  (let [field (:field item)]
    (cond
      (date-field? field) (prepare-date item)
      (translatable-field? field) (prepare-translate lang item)
      :else item)))

(defn prepare-return-data
  "Prepares some fields and translates others"
  [lang result]
  (update result :data #(map (partial prepare-item lang) %)))

(defn run
  "List users report by query"
  [{:keys [user-changes-report-gateway] :as context} data]
  (go
    (let [query (prepare-query context data)
          lang (or (:lang data) "en")
          auth-ch (auth/authorized? (assoc context :action :reports-access))
          query-ch (urg/query user-changes-report-gateway context query)]
      (metrics/report-generated "user-changes" data)
      (result/enforce-let [auth-result (<! auth-ch)
                           query-result (<! query-ch)]
        (prepare-return-data lang query-result)))))
