(ns clanhr.reports-api.core.generate-excel
  "Generates an excel document"
  (:refer-clojure :exclude [run!])
  (:import
   (java.io FileOutputStream FileInputStream InputStream)
   (java.util Date Calendar)
   (org.apache.poi.xssf.usermodel XSSFWorkbook))
  (:require [result.core :as result]
            [clojure.core :as core]
            [clojure.java.io :as io]
            [dk.ative.docjure.spreadsheet :as spreadsheet]
            [clojure.core.async :refer [go go-loop <!]]))

(defn- mkdirp [path]
  (let [dir (java.io.File. path)]
    (if (.exists dir)
      true
      (.mkdirs dir))))

(mkdirp "tmp")

(def header-style {:background :grey_25_percent
                   :font {:bold true}})

(defn- set-header-style
  "Sets the style of first row (header) of the sheet"
  ([workbook sheet]
   (if-let [header-row (first (spreadsheet/row-seq sheet))]
     (set-header-style workbook sheet header-row)))
  ([workbook sheet header-row]
   (spreadsheet/set-row-style! header-row
                               (spreadsheet/create-cell-style! workbook header-style))))

(defn run
  "Returns a stream that will have the content"
  [context params]
  (let [sheet-name (or (:sheet-name params) "Sheet")
        file-name (str (gensym) ".xlsx")
        file-path (-> (io/file "tmp" file-name) .getPath)
        data (:data params)
        workbook (spreadsheet/create-workbook sheet-name data)
        sheet (spreadsheet/select-sheet sheet-name workbook)
        header-row-number (or (:header-row-number params) 0)
        header-row (nth (spreadsheet/row-seq sheet) header-row-number)]
    (when header-row
      (set-header-style workbook sheet header-row))
    (dotimes [n (count (first data))]
      (.autoSizeColumn sheet n))
    (spreadsheet/save-workbook! file-path workbook)
    (result/success {:file-name file-path})))

(defn multisheet-run
  "Builds and excel file with multiple sheets. Expects a params map with a
  sheets key that maps sheet name to data."
  [context params]
  (let [file-name (str (gensym) ".xlsx")
        file-path (-> (io/file "tmp" file-name) .getPath)
        workbook (XSSFWorkbook.)]
    (core/run! (fn [[sheet-name sheet-data]]
                 (let [sheet (spreadsheet/add-sheet! workbook sheet-name)]
                   (spreadsheet/add-rows! sheet sheet-data)
                   (set-header-style workbook sheet)
                   (dotimes [n (count (first sheet-data))]
                     (.autoSizeColumn sheet n))))
          (:sheets params))
    (spreadsheet/save-workbook! file-path workbook)
    (result/success {:file-name file-path})))
