(ns clanhr.reports-api.controllers.user-kms
  (:require [clanhr.reply.core :as reply]
            [clojure.string :as s]
            [clanhr.reports-api.core.user-kms :as user-kms]
            [result.core :as result]
            [clanhr.i18n.core :as i18n]
            [clj-time.coerce :as c]
            [clanhr.reports-api.core.get-user :as get-user]
            [clanhr.reports-api.core.get-account-settings :as get-account-settings]
            [clojure.core.async :refer [<! thread]]
            [clanhr.reports-api.core.generate-excel :as generate-excel]
            [clanhr.reports-api.lib.utils :as utils]))

(defn get-total
  [result]
  (->> result
       :data
       (map #(utils/select-values % [:total-with-tax]))
       flatten
       (reduce +)))

(defn prepare-excel-data
  "Prepares the excel data"
  [lang result user account start-date end-date]
  (let [footer [["" "" "" "" "" "" "" (utils/get-name lang :total) (get-total result)]
                []
                [(utils/get-name lang :date) "" "" (utils/get-name lang :signature)]
                []
                []
                [(utils/get-name lang :company-name) (get-in account [:billing-data :company-name])]
                [(utils/get-name lang :company-address) (utils/format-address (get-in account [:billing-data :address]))]
                [(utils/get-name lang :company-vat-number) (get-in account [:billing-data :vat-number])]]]
    {:header-row-number 7
     :data (into
             (into [[(i18n/t lang :user-kms-report-title)]

                    []

                    [(utils/get-name lang :name) (get-in user [:personal-data :name]) "" (utils/get-name lang :interval) (utils/format-interval start-date end-date)]
                    [(utils/get-name lang :colaborator-id) (get-in user [:company-data :colaborator-id])]
                    [(utils/get-name lang :cost-centers) (s/join ", " (get-in user [:company-data :cost-centers]))]
                    [(utils/get-name lang :license-plate) (get-in user [:personal-data :license-plate])]

                    []

                    [(i18n/t lang :date)
                     (i18n/t lang :reason)
                     (i18n/t lang :state)
                     (i18n/t lang :origin)
                     (i18n/t lang :destination)
                     (i18n/t lang :license-plate)
                     (i18n/t lang :value-per-km)
                     (i18n/t lang :number-of-kms)
                     (i18n/t lang :total-with-tax)]]
                   (mapv (fn [item]
                           [(c/to-date (:km-date item))
                            (:reason item)
                            (i18n/t lang (:state item))
                            (:origin item)
                            (:destination item)
                            (:license-plate item)
                            (:value-per-km item)
                            (:number-of-kms item)
                            (:total-with-tax item)])
                         (:data result)))
             footer)}))

(defn prepare-params
  "Builds params as expected by the interactor"
  [params]
  (-> params
      (assoc :status (s/split (or (:status params) "") #","))))

(defn excel
  "List kms expenses report in excel file"
  [request]
  (reply/async-reply
    (let [context (:context request)
          params (prepare-params (:params request))
          lang (or (:lang params "en"))]
      (result/enforce-let [result (<! (user-kms/run (:context request) params))
                           user (<! (get-user/run (:context request) (:user-id params)))
                           account (<! (get-account-settings/run (:context request)))
                           excel (<! (thread (generate-excel/run context (prepare-excel-data lang result user account (:start-date params) (:end-date params)))))]
        (let [file-name (:file-name excel)]
          (reply/excel-file file-name (i18n/t lang :user-kms-excel-file "Kms.xlsx")))))))

(defn handler
  "List expenses report"
  [request]
  (reply/async-result
    (user-kms/run (:context request)
                  (prepare-params (:params request)))))
