(ns clanhr.reports-api.controllers.user-expenses
  (:require [clanhr.reply.core :as reply]
            [clanhr.i18n.core :as i18n]
            [result.core :as result]
            [clj-time.coerce :as c]
            [clj-time.format :as f]
            [clojure.string :as s]
            [clanhr.reports-api.core.get-user :as get-user]
            [clanhr.reports-api.core.get-account-settings :as get-account-settings]
            [clojure.core.async :refer [<! thread]]
            [clanhr.reports-api.core.generate-excel :as generate-excel]
            [clanhr.reports-api.core.user-expenses :as user-expenses]
            [clanhr.reports-api.lib.utils :as utils]))

(defn get-total
  [result]
  (->> result
       :data
       (map #(utils/select-values % [:value]))
       flatten
       (map (comp read-string #(s/replace % "," ".")))
       (reduce +)))

(defn prepare-excel-data
  "Prepares the excel data"
  [lang result account user start-date end-date]
  (let [footer [["" "" "" "" (i18n/t lang :total) (get-total result)]
                []
                [(utils/get-name lang :date) "" "" (utils/get-name lang :signature)]
                []
                []
                [(utils/get-name lang :company-name) (get-in account [:billing-data :company-name])]
                [(utils/get-name lang :company-address) (utils/format-address (get-in account [:billing-data :address]))]
                [(utils/get-name lang :company-vat-number) (get-in account [:billing-data :vat-number])]]]
    {:header-row-number 7
     :data (into
             (into [[(i18n/t lang :user-expenses-report-title)]

                    []

                    [(utils/get-name lang :name) (get-in user [:personal-data :name])]
                    [(utils/get-name lang :colaborator-id) (get-in user [:company-data :colaborator-id])]
                    [(utils/get-name lang :cost-centers) (s/join ", " (get-in user [:company-data :cost-centers]))]
                    [(utils/get-name lang :interval) (utils/format-interval start-date end-date)]

                    []

                    [(i18n/t lang :date)
                     (i18n/t lang :state)
                     (i18n/t lang :type)
                     (i18n/t lang :description)
                     (i18n/t lang :document-number)
                     (i18n/t lang :value)]]
                   (mapv (fn [item]
                           [(c/to-date (:expense-date item))
                            (i18n/t lang (:state item))
                            (i18n/t lang (:type item))
                            (:description item)
                            (:document-number item)
                            (s/replace (:value item) "," ".")])
                         (:data result)))
             footer)}))

(defn prepare-params
  "Builds params as expected by the interactor"
  [params]
  (-> params
      (assoc :status (clojure.string/split (or (:status params) "") #","))
      (assoc :types (clojure.string/split (or (:types params) "") #","))))

(defn excel
  "List expenses report in excel file"
  [request]
  (reply/async-reply
    (let [context (:context request)
          params (prepare-params (:params request))
          lang (or (:lang params "en"))]
      (result/enforce-let [result (<! (user-expenses/run (:context request) params))
                           user (<! (get-user/run (:context request) (:user-id params)))
                           account (<! (get-account-settings/run (:context request)))
                           excel (<! (thread (generate-excel/run context (prepare-excel-data lang result account user (:start-date params) (:end-date params)))))]
        (let [file-name (:file-name excel)]
          (reply/excel-file file-name (i18n/t lang :user-expenses-excel-file "Expenses.xlsx")))))))

(defn handler
  "List expenses report"
  [request]
  (reply/async-result
    (user-expenses/run (:context request)
                       (prepare-params (:params request)))))
