(ns clanhr.reports-api.controllers.routes
  (:gen-class)
  (:require [clojure.stacktrace]
            [new-reliquary.ring :refer [wrap-prd-newrelic-transaction]]
            [ring.middleware.json :refer [wrap-json-response]]
            [compojure.handler :as handler]
            [clanhr.auth.auth-middleware :as auth]
            [clanhr.logger.middleware :as logger-middleware]
            [com.stuartsierra.component :as component]
            [clanhr.analytics.errors :as errors]
            [clanhr.analytics.metrics :as metrics]
            [clanhr.reports-api.models.user-changes :as user-changes-model]
            [clanhr.reports-api.models.expense :as expenses-model]
            [clanhr.reports-api.models.kms :as km-model]
            [clanhr.reports-api.models.absence-user :as absence-user-model]
            [clanhr.reports-api.models.directory-user :as directory-user-model]
            [clanhr.reports-api.models.vacations-and-absences :as vacations-and-absences-model]
            [clanhr.reports-api.models.vacations-balance :as vacations-balance-model]
            [clanhr.reports-api.controllers.healthcheck :as healthcheck]
            [clanhr.reports-api.controllers.list-vacations-and-absences :as list-vacations-and-absences]
            [clanhr.reports-api.controllers.list-timelogs :as list-timelogs]
            [clanhr.reports-api.controllers.sample-excel :as sample-excel]
            [clanhr.reports-api.controllers.setup-vacations :as setup-vacations]
            [clanhr.reports-api.controllers.list-kms :as list-kms]
            [clanhr.reports-api.controllers.user-kms :as user-kms]
            [clanhr.reports-api.controllers.user-expenses :as user-expenses]
            [clanhr.reports-api.controllers.register-expense :as register-expenses]
            [clanhr.reports-api.controllers.user-documents :as user-documents]
            [clanhr.reports-api.controllers.list-vacations-balance :as list-vacations-balance]
            [clanhr.reports-api.controllers.list-users :as list-users]
            [clanhr.reports-api.controllers.list-user-changes :as list-user-changes]
            [clanhr.reports-api.controllers.sync-absence-data :as sync-absence-data]
            [clanhr.reports-api.controllers.register-timelog :as register-timelog]
            [clanhr.reports-api.controllers.register-kms :as register-kms]
            [clanhr.reports-api.controllers.register-document :as register-document]
            [clanhr.reports-api.controllers.sync-user :as sync-user]
            [clanhr.reports-api.controllers.delete-user :as delete-user]
            [clanhr.reports-api.middlewares.context :as context]
            [schema.core :as s]
            [aleph.http :as http]
            [clanhr.reply.core :as reply]
            [clojure.core.async :as a]
            [ring.middleware.params :as ring-params]
            [ring.middleware.cors :as cors]
            [compojure.core :as core :refer [GET POST PUT DELETE defroutes]]
            [compojure.api.sweet :as sweet]
            [compojure.route :as route]))

(def sweet-public-routes
  (sweet/routes
    (sweet/GET "/" request
               :return {:name s/Str}
               :tags ["utilities"]
               :no-doc true
               :summary "The same as /healthcheck"
               (healthcheck/handler request))
    (sweet/GET "/sample-excel" request
               :tags ["utilities"]
               :summary "Returns a sample excel"
               (sample-excel/handler request))
    (sweet/GET "/healthcheck" request
               :return {:name s/Str}
               :tags ["utilities"]
               :summary "Just a hello world endpoint"
               (healthcheck/handler request))))

(defn auth-middleware
  "The middleware collection of functions that will run for authed routes"
  [system]
  (comp auth/run
        #(logger-middleware/run % :reports-api)
        #(context/run % system)))

(def default-responses
  "The default error responses that services usually use"
  {400 {:description "Request information/format is invalid."
                                :schema {:success (s/eq false)
                                         (s/optional-key :data) s/Any}}})

(defn sweet-private-routes
  "Declares and defines routes that need auth token"
  [system middleware]
  (sweet/routes
    (sweet/GET "/list-users" request
               :query-params [{page :-  s/Int 1}
                              {per-page :- s/Int 10}
                              {name :- s/Str nil}
                              {position :- s/Str nil}
                              {teams :- s/Str nil}
                              {projects :- s/Str nil}
                              {tags :- s/Str nil}
                              {approver-id :- s/Uuid nil}]
               :middleware [middleware]
               :return {:success s/Bool
                        :total-results s/Int
                        :data [directory-user-model/schema]
                        :number-of-pages s/Int
                        :current-page s/Int}
               :responses default-responses
               :tags ["users" "json"]
               :summary "Users report data"
               :description "Returns a paginated view of the users on the system.
                            This endpoint is used for the user reports page."
               (list-users/handler request))

    (sweet/GET "/list-users.xlsx" request
               :query-params [{page :-  s/Int 1}
                              {per-page :- s/Int 10}
                              {name :- s/Str nil}
                              {position :- s/Str nil}
                              {teams :- s/Str nil}
                              {projects :- s/Str nil}
                              {tags :- s/Str nil}
                              {approver-id :- s/Uuid nil}]
               :middleware [middleware]
               :responses default-responses
               :tags ["users" "excel"]
               :summary "Users excel data"
               :description "Returns the excel file for the given query data"
               (list-users/excel request))

    (sweet/GET "/list-user-changes" request
               :query-params [{page :-  s/Int 1}
                              {per-page :- s/Int 10}
                              {name :- s/Str nil}
                              {position :- s/Str nil}
                              {teams :- s/Str nil}
                              {projects :- s/Str nil}
                              {tags :- s/Str nil}
                              {start-date :- s/Str nil}
                              {end-date :- s/Str nil}
                              {approver-id :- s/Uuid nil}]
               :middleware [middleware]
               :return {:success s/Bool
                        :data [user-changes-model/schema]
                        :total-results s/Int
                        :number-of-pages s/Int
                        :current-page s/Int}
               :responses default-responses
               :tags ["user-changes" "json"]
               :summary "User changes report data"
               :description "Returns a paginated view of the changes that were
                            made to the users model."
               (list-user-changes/handler request))

    (sweet/GET "/list-user-changes.xlsx" request
               :query-params [{page :-  s/Int 1}
                              {per-page :- s/Int 10}
                              {name :- s/Str nil}
                              {position :- s/Str nil}
                              {teams :- s/Str nil}
                              {projects :- s/Str nil}
                              {tags :- s/Str nil}
                              {start-date :- s/Str nil}
                              {end-date :- s/Str nil}
                              {approver-id :- s/Uuid nil}]
               :middleware [middleware]
               :responses default-responses
               :tags ["user-changes" "excel"]
               :summary "User changes excel data"
               :description "Returns the excel file for the given query data"
               (list-user-changes/excel request))

    (sweet/GET "/list-vacations-balance" request
               :query-params [{page :-  s/Int 1}
                              {per-page :- s/Int 10}
                              {name :- s/Str nil}
                              {teams :- s/Str nil}
                              {projects :- s/Str nil}
                              {tags :- s/Str nil}
                              {year :- s/Int nil}]
               :middleware [middleware]
               :return {:success s/Bool
                        :data [vacations-balance-model/schema]
                        :total-results s/Int
                        :number-of-pages s/Int
                        :current-page s/Int}
               :responses default-responses
               :tags ["vacations-balance" "json"]
               :summary "Vacations balance report data"
               :description "Returns a paginated view of the vacations balance
                            for each user."
               (list-vacations-balance/handler request))

    (sweet/GET "/list-vacations-balance.xlsx" request
               :query-params [{page :-  s/Int 1}
                              {per-page :- s/Int 10}
                              {name :- s/Str nil}
                              {teams :- s/Str nil}
                              {projects :- s/Str nil}
                              {tags :- s/Str nil}
                              {year :- s/Int nil}]
               :middleware [middleware]
               :responses default-responses
               :tags ["vacations-balance" "excel"]
               :summary "Vacations balance excel data"
               :description "Returns the excel file for the given query data"
               (list-vacations-balance/excel request))

    (sweet/GET "/user-expenses" request
               :query-params [{user-id :- s/Str nil}
                              {lang :- s/Str "en"}
                              {status :- [s/Str] nil}
                              {types  :- [s/Str] nil}
                              {start-date   :- s/Str nil}
                              {end-date   :- s/Str nil}]
               :middleware [middleware]
               :return {:success s/Bool
                        :data [expenses-model/schema]
                        :total-results s/Int
                        :current-page s/Int}
               :tags ["expenses" "json"]
               :summary "Expenses report data"
               :description "Returns a list of expenses given query data"
               (user-expenses/handler request))

    (sweet/GET "/user-expenses.xlsx" request
               :query-params [{user-id :- s/Str nil}
                              {status :- [s/Str] nil}
                              {lang :- s/Str "en"}
                              {types  :- [s/Str] nil}
                              {start-date   :- s/Str nil}
                              {end-date   :- s/Str nil}]
               :middleware [middleware]
               :responses default-responses
               :tags ["expenses" "excel"]
               :summary "Expenses an excel file report data"
               :description "Returns the excel file for the given query data"
               (user-expenses/excel request))

    (sweet/GET "/user-kms" request
               :query-params [{user-id :- s/Str nil}
                              {lang :- s/Str "en"}
                              {status :- [s/Str] nil}
                              {start-date   :- s/Str nil}
                              {end-date   :- s/Str nil}]
               :middleware [middleware]
               :return {:success s/Bool
                        :data s/Any
                        :total-results s/Int
                        :current-page s/Int}
               :tags ["expenses" "kms" "json"]
               :summary "Km expenses report data"
               :description "Returns a list of km expenses given query data"
               (user-kms/handler request))

    (sweet/GET "/user-kms.xlsx" request
               :query-params [{user-id :- s/Str nil}
                              {status :- [s/Str] nil}
                              {lang :- s/Str "en"}
                              {start-date   :- s/Str nil}
                              {end-date   :- s/Str nil}]
               :middleware [middleware]
               :responses default-responses
               :tags ["expenses" "kms" "excel"]
               :summary "Km expenses an excel file report data"
               :description "Returns the excel file for the given query data"
               (user-kms/excel request))

    (sweet/GET "/list-vacations-and-absences" request
               :query-params [{page :-  s/Int 1}
                              {per-page :- s/Int 10}
                              {name :- s/Str nil}
                              {types :- s/Str nil}
                              {start-date :- s/Str nil}
                              {end-date :- s/Str nil}
                              {teams :- s/Str nil}
                              {projects :- s/Str nil}
                              {tags :- s/Str nil}
                              {year :- s/Int nil}]
               :middleware [middleware]
               :return {:success s/Bool
                        :data [vacations-and-absences-model/schema]
                        :total-results s/Int
                        :number-of-pages s/Int
                        :current-page s/Int}
               :responses default-responses
               :tags ["vacations-and-absences" "json"]
               :summary "Vacations and absences report data"
               :description "Returns a paginated view of the vacations and
                            absences for each user."
               (list-vacations-and-absences/handler request))

    (sweet/GET "/list-vacations-and-absences.xlsx" request
               :query-params [{page :-  s/Int 1}
                              {per-page :- s/Int 10}
                              {name :- s/Str nil}
                              {types :- s/Str nil}
                              {start-date :- s/Str nil}
                              {end-date :- s/Str nil}
                              {teams :- s/Str nil}
                              {projects :- s/Str nil}
                              {tags :- s/Str nil}
                              {year :- s/Int nil}]
               :middleware [middleware]
               :responses default-responses
               :tags ["vacations-and-absences" "excel"]
               :summary "Vacations and absences excel data"
               :description "Returns the excel file for the given query data"
               (list-vacations-and-absences/excel request))

    (comment
    (sweet/POST "/setup-vacations" request
               :middleware [middleware]
               :body [absence-user (s/maybe absence-user-model/schema)]
               :return {:success s/Bool :data s/Any}
               :responses default-responses
               :tags ["sync"]
               :summary "Sync vacations"
               :description "Gets the vacations information for a user and updates related reports."
               (setup-vacations/handler request))
    )

    (sweet/PUT "/sync-user" request
               :middleware [middleware]
               :body [directory-user (s/maybe directory-user-model/schema)]
               :return {:success s/Bool
                        :absence-report s/Any
                        :timelog s/Any
                        :user-report s/Any
                        :user-changes-report s/Any
                        :vacations-balance s/Any}
               :responses default-responses
               :tags ["sync"]
               :summary "Sync user"
               :description "With the directory user, updates all reports that
                            have information on the given user."
               (sync-user/handler request))

    (sweet/POST "/delete-user/:user-id" request
               :middleware [middleware]
               :path-params [user-id :- s/Uuid]
               :return {:success s/Bool}
               :responses default-responses
               :tags ["sync"]
               :summary "Deletes user information"
               :description "Deletes all user information from all reports"
               (delete-user/handler request))))

(defroutes private-routes
  (GET "/list-timelogs" request (list-timelogs/handler request))
  (PUT "/sync-absence-data" request (sync-absence-data/handler request))
  (POST "/setup-vacations" request (setup-vacations/handler request))
  (POST "/register-timelog" request (register-timelog/handler request))
  (POST "/register-kms" request (register-kms/handler request))
  (POST "/register-expenses" request (register-expenses/handler request))
  (POST "/register-document" request (register-document/handler request))
  (GET "/user-documents/:user-id/:account-id" request (user-documents/handler request))
  (route/not-found (reply/not-found {:success false :data "handler-not-found"})))

(defn- register-error-500
  "Some middleware can return the response as a 500 error response. If so,
  let's log it properly"
  [req res]
  (let [body (slurp (:body res))
        ex (ex-info "Error 500 in middleware" {:data body})]
    (errors/request-exception ex req)
    (reply/exception ex)))

(defn- wrap-exception-handler
  [handler]
  (fn [req]
    (try
      (let [res (handler req)]
        (if (= 500 (:status res))
          (register-error-500 req res)
          res))
      (catch Exception e
        (errors/request-exception e req)
        (reply/exception e)))))

(defn- setup-cors
  "Setup cors"
  [handler]
  (cors/wrap-cors handler
                  :access-control-allow-origin
                  [#"^https?://(.*)clanhr.com(.*)"
                   #"^https?://clanhr(.*)cloudapp.net(.*)"
                   #"^http://clanhr-repack-frontend.herokuapp.com(.*)"
                   #"^http://clanhr-repack-frontend-2.herokuapp.com(.*)"
                   #"^http://localhost(.*)"
                   #"^http://[0-9]+(?:\.[0-9]+){3}(:[0-9]+)?"
                   ]
                  :access-control-allow-methods [:get :put :post :delete]))

(defn routes [system]
  (let [auth-middleware (auth-middleware system)]
    (sweet/api
      {:swagger {:ui "/api-docs"
                 :spec "/swagger.json"
                 :options {:ui {:jsonEditor true}}
                 :data {:produces ["application/json"],
                        :consumes ["application/json"],
                        :info {:title "ClanHR Reports API"
                               :description "This API has the endpoints used by the UI
                                            to generate the reports page and to export
                                            reports to excel. It also has some sync
                                            endpoints, used internally by other services."
                               :contact {:name "ClanHR API Team"
                                         :email "hello@clanhr.com"
                                         :url "http://www.clanhr.com"}}
                        :tags [{:name "utilities" :description "some utility endpoints"}
                               {:name "sync" :description "endpoints that update reports"}
                               {:name "json" :description "endpoints that provide reports data as json"}
                               {:name "excel" :description "endpoints that provide reports data as excel"}
                               {:name "vacations-balance" :description "user vacations balance reports access"}
                               {:name "vacations-and-absences" :description "user vacations and absences reports access"}
                               {:name "user-changes" :description "user changes reports access"}
                               {:name "users" :description "reports access"}]}}}
      (sweet/routes
        sweet-public-routes
        (sweet-private-routes system auth-middleware)
        (-> (handler/api private-routes)
            (logger-middleware/run :reports-api)
            (context/run system)
            (auth/run))))))

(defn app
  "Creates the app pipeline"
  [system]
  (-> (routes system)
      (compojure.handler/api)
      (wrap-exception-handler)
      (setup-cors)
      (metrics/http-request-metric-fn "reports-api")
      (wrap-json-response)
      (wrap-prd-newrelic-transaction)))

(defn- get-port
  "Gets the port to run the service"
  []
  (or (get (System/getenv) "PORT")
      (get (System/getenv) "CLANHR_REPORTS_API_PORT")
      "5000"))

(defn- on-shutdown
  "Runs shutdown hooks"
  [system]
  (component/stop system)
  (println "Shutdown system..."))

(defn -main [& args]
  (let [port (Integer/parseInt (get-port))
        system (-> (context/system) (component/start))]
    (println (str "** Reports API " (or (get (System/getenv) "CLANHR_ENV" "development"))
                  " running on port " port))
    (.addShutdownHook (Runtime/getRuntime) (Thread. (partial on-shutdown system)))
    (http/start-server (app system) {:port port :executor :none})
    (a/<!! (a/chan))))
