(ns clanhr.reports-api.controllers.list-vacations-balance
  (:require [clanhr.reply.core :as reply]
            [clanhr.reply.json :as json]
            [clanhr.i18n.core :as i18n]
            [result.core :as result]
            [clojure.core.async :refer [go go-loop <! thread]]
            [clanhr.reports-api.core.list-vacations-balance :as list-vacations-balance]
            [clanhr.reports-api.core.generate-excel :as generate-excel]))

(defn prepare-params
  "Builds params as expected by the interactor"
  [params]
  (-> params
      (assoc :tags (clojure.string/split (or (:tags params) "") #","))
      (assoc :teams (clojure.string/split (or (:teams params) "") #","))
      (assoc :projects (clojure.string/split (or (:projects params) "") #","))))

(defn prepare-excel-data
  "Prepares the excel data to be a simple table"
  [lang result]
  {:data (into [[(i18n/t lang :colaborator-id)
                 (i18n/t lang :name)
                 (i18n/t lang :totalDays)
                 (i18n/t lang :enjoyedDays)
                 (i18n/t lang :scheduledDays)
                 (i18n/t lang :freeDays)]]
           (mapv (fn [item]
                         [(:colaborator_id item)
                          (:name item)
                          (:total_days item)
                          (:enjoyed_days item)
                          (:scheduled_days item)
                          (- (:total_days item) (:enjoyed_days item) (:scheduled_days item))])
                       (:data result)))})

(defn handler
  "List vacation balances report"
  [request]
  (reply/async-result
    (list-vacations-balance/run! (:context request)
                                 (prepare-params (:params request)))))

(defn excel
  "Vacations and absences excel report"
  [request]
  (reply/async-reply
    (let [context (:context request)
          params (-> (:params request)
                     (assoc :report-metric-source "excel")
                     (assoc :per-page 10000))
          lang (or (:lang params) "en")]
      (result/enforce-let [result (<! (list-vacations-balance/run! context params))
                           excel (<! (thread (generate-excel/run context (prepare-excel-data lang result))))]
        (let [file-name (:file-name excel)]
          (reply/excel-file file-name (i18n/t lang :vacations-balance-excel-file "VacationsBalance.xlsx")))))))
