(ns clanhr.reports-api.controllers.list-vacations-and-absences
  (:require [clanhr.reply.core :as reply]
            [clanhr.reply.json :as json]
            [clanhr.i18n.core :as i18n]
            [result.core :as result]
            [clj-time.coerce :as c]
            [clojure.core.async :refer [go go-loop <! thread]]
            [clanhr.reports-api.core.list-vacations-and-absences :as list-vacations-and-absences]
            [clanhr.reports-api.core.generate-excel :as generate-excel]))

(defn handler
  "List vacations and absences report"
  [request]
  (reply/async-result
    (list-vacations-and-absences/run! (:context request)
                                      (:params request))))

(defn- get-duration
  "Gets the duration of an absence item"
  [item]
  (if-let [duration (:duration item)]
    duration
    (if (= "partial-day" (:duration-type item))
      0.5M)))

(defn- get-duration-type-excel
  "Gets the duration type to use on exel"
  [item]
  (cond
    (= "partial-day" (:duration-type item)) "days"
    :else (:duration-type item)))

(defn prepare-excel-data
  "Prepares the excel data to be a simple table"
  [lang result]
  {:header-row-number 0
   :data (into [[(i18n/t lang :colaborator-id)
                 (i18n/t lang :name)
                 (i18n/t lang :type)
                 (i18n/t lang :start-date)
                 (i18n/t lang :end-date)
                 (i18n/t lang :total)
                 (i18n/t lang :unit)]]
           (mapv (fn [item]
                   [(:colaborator_id item)
                    (:name item)
                    (i18n/t lang
                            (str "absence-" (:absence-type item))
                            (:absence-type item))
                    (c/to-date (or (:start_date item) (:start-date item)))
                    (c/to-date (or (:end_date item) (:end-date item)))
                    (get-duration item)
                    (i18n/t lang (get-duration-type-excel item))])
                 (:data result)))})

(defn excel
  "Vacations and absences excel report"
  [request]
  (reply/async-reply
    (let [context (:context request)
          params (-> (:params request)
                     (assoc :report-metric-source "excel")
                     (assoc :per-page 10000))
          lang (or (:lang params) "en")]
      (result/enforce-let [result (<! (list-vacations-and-absences/run! context params))
                           excel (<! (thread (generate-excel/run context (prepare-excel-data lang result))))]
        (let [file-name (:file-name excel)]
          (reply/excel-file file-name (i18n/t lang :vacations-and-absences-excel-file "Absences.xlsx")))))))
