(ns clanhr.reports-api.controllers.list-users
  (:require [clanhr.reply.core :as reply]
            [clanhr.reply.json :as json]
            [clanhr.i18n.core :as i18n]
            [result.core :as result]
            [clojure.string :as string]
            [clj-time.coerce :as c]
            [clojure.core.async :refer [go go-loop <! thread]]
            [clanhr.reports-api.core.list-users :as list-users]
            [clanhr.reports-api.core.generate-excel :as generate-excel]))

(defn handler
  "List users report"
  [request]
  (reply/async-result
    (list-users/run (:context request) (:params request))))

(defn- str-coll
  "Renders an str collection as one string"
  [coll]
  (when coll
    (string/join ", " coll)))

(defn- personal-header-row
  "Gets the reader row for the given language"
  [lang]
  [(i18n/t lang :colaborator-id)
   ;; personal data
   (i18n/t lang :name)
   (i18n/t lang :birth-date)
   (i18n/t lang :civil-status)
   (i18n/t lang :nationality)

   ;; contacts
   (i18n/t lang :personal-email)
   (i18n/t lang :personal-phone)

   (i18n/t lang :address.street)
   (i18n/t lang :address.parish)
   (i18n/t lang :address.district)
   (i18n/t lang :address.zip-code)
   (i18n/t lang :address.city)
   (i18n/t lang :address.country-code)

   (i18n/t lang :fiscal-address.street)
   (i18n/t lang :fiscal-address.parish)
   (i18n/t lang :fiscal-address.district)
   (i18n/t lang :fiscal-address.zip-code)
   (i18n/t lang :fiscal-address.city)
   (i18n/t lang :fiscal-address.country-code)

   ;; fiscal information
   (i18n/t lang :income-ownership)
   (i18n/t lang :number-of-dependents)
   (i18n/t lang :handicap)

   ;; documents
   (i18n/t lang :idcard)
   (i18n/t lang :social-security-number)
   (i18n/t lang :vat-number)
   (i18n/t lang :other-document)

   ;; bank data
   (i18n/t lang :bank-account.bank-name)
   (i18n/t lang :bank-account.swift)
   (i18n/t lang :bank-account.iban)

   ;; school
   (i18n/t lang :degree)
   (i18n/t lang :school.school-course)
   (i18n/t lang :school.school-name)

   ;; about me
   (i18n/t lang :about-me)
   (i18n/t lang :facebook)
   (i18n/t lang :twitter)
   (i18n/t lang :linkedin)

   ;; emergency contact
   (i18n/t lang :emergency-contact.contact-name)
   (i18n/t lang :emergency-contact.phone)
   (i18n/t lang :emergency-contact.relationship)])

(defn- company-header-row
  "Gets the reader row for the given language"
  [lang]
  [(i18n/t lang :colaborator-id)
   (i18n/t lang :name)

   (i18n/t lang :cost-center)
   (i18n/t lang :professional-email)
   (i18n/t lang :professional-phone)

   ;; salary info
   (i18n/t lang :salary)
   (i18n/t lang :food-allowance)
   (i18n/t lang :iht)
   (i18n/t lang :expenses)
   (i18n/t lang :salary-observations)

   ;; job info
   (i18n/t lang :tags)
   (i18n/t lang :job-position)
   (i18n/t lang :approver)
   (i18n/t lang :job-description)

   ;; contract
   (i18n/t lang :contract-type)
   (i18n/t lang :contract-start-date)
   (i18n/t lang :contract-end-date)
   (i18n/t lang :contract-observations)])

(defn- get-current-contract
  "Gets the current contrat of the user"
  [user]
  (or (get-in user [:company-data :current-contract])
      (last (get-in user [:company-data :contracts]))))

(defn- personal-user-row
  "Gets the data row for a user"
  [lang item]
  (let [contract (get-current-contract item)]
    [(get-in item [:company-data :colaborator-id])

     ;; personal data
     (get-in item [:personal-data :name])
     (c/to-date (get-in item [:personal-data :birth-date]))
     (i18n/t lang (get-in item [:personal-data :civil-status]))
     (get-in item [:personal-data :nationality])

     ;; contacts
     (get-in item [:personal-data :email])
     (get-in item [:personal-data :phone])

     (get-in item [:personal-data :address :street])
     (get-in item [:personal-data :address :parish])
     (get-in item [:personal-data :address :district])
     (get-in item [:personal-data :address :zip-code])
     (get-in item [:personal-data :address :city])
     (get-in item [:personal-data :address :country-code])

     (get-in item [:personal-data :fiscal-address :street])
     (get-in item [:personal-data :fiscal-address :parish])
     (get-in item [:personal-data :fiscal-address :district])
     (get-in item [:personal-data :fiscal-address :zip-code])
     (get-in item [:personal-data :fiscal-address :city])
     (get-in item [:personal-data :fiscal-address :country-code])

     ;; fiscal information
     (i18n/t lang (get-in item [:personal-data :income-ownership]))
     (get-in item [:personal-data :dependents])
     (i18n/t lang (get-in item [:personal-data :deficients]))

     ;; documents
     (get-in item [:personal-data :idcard])
     (get-in item [:personal-data :social-security-number])
     (get-in item [:personal-data :vat-number])
     (get-in item [:personal-data :other-document])

     ;; bank data
     (get-in item [:personal-data :bank-account :bank-name])
     (get-in item [:personal-data :bank-account :swift])
     (get-in item [:personal-data :bank-account :iban])

     ;; school name
     (i18n/t lang (get-in item [:personal-data :school :school-degree]))
     (get-in item [:personal-data :school :course])
     (get-in item [:personal-data :school :school-name])

     ;; about me
     (get-in item [:personal-data :about-me])
     (get-in item [:personal-data :facebook])
     (get-in item [:personal-data :twitter])
     (get-in item [:personal-data :linkedin])

     ;; emergency contact
     (get-in item [:personal-data :emergency-contact :contact-name])
     (get-in item [:personal-data :emergency-contact :phone])
     (get-in item [:personal-data :emergency-contact :relationship])]))

(defn- company-user-row
  "Gets the data row for a user"
  [lang item]
  (let [contract (get-in item [:company-data :current-contract])]
    [(get-in item [:company-data :colaborator-id])
     (get-in item [:personal-data :name])

     (str-coll (get-in item [:company-data :cost-centers]))
     (get-in item [:company-data :email])
     (get-in item [:company-data :phone])

     ;; salary info
     (get-in item [:company-data :salary])
     (get-in item [:company-data :food-allowance])
     (get-in item [:company-data :iht])
     (get-in item [:company-data :expenses])
     (get-in item [:company-data :salary-observations])

     ;; job info
     (str-coll (get-in item [:company-data :tags]))
     (get-in item [:company-data :position])
     (get-in item [:company-data :approver-name])
     (get-in item [:company-data :job-description])

     ;; contract info
     (i18n/t lang (get-in contract [:contract-type]))
     (c/to-date (get-in contract [:contract-start-date]))
     (c/to-date (get-in contract [:contract-end-date]))
     (get-in contract [:contract-observations])]))

(defn- build-sheet-data
  "Creates a proper data to be consumed by the sheet"
  [lang result header-row-fn user-row-fn]
  (into [(header-row-fn lang)]
        (mapv #(user-row-fn lang %) (:data result))))

(defn prepare-excel-data
  "Prepares the excel data to be a simple table"
  [lang result]
  (let [personal-data (build-sheet-data lang result personal-header-row personal-user-row)
        company-data (build-sheet-data lang result company-header-row company-user-row)]
    {:sheets {(i18n/t lang "personal-data" "Personal data") personal-data
              (i18n/t lang "company-data" "Company data") company-data}}))

(defn excel
  "Users excel report"
  [request]
  (reply/async-reply
    (let [context (:context request)
          params (-> (:params request)
                     (assoc :report-metric-source "excel")
                     (assoc :per-page 10000))
          lang (or (:lang params) "en")]
      (result/enforce-let [result (<! (list-users/run context params))
                           excel (<! (thread (generate-excel/multisheet-run context (prepare-excel-data lang result))))]
        (let [file-name (:file-name excel)]
          (reply/excel-file file-name (i18n/t lang :users-excel-file "Users.xlsx")))))))
