(ns clanhr.auth.core
  "This namespace has some utilities to deal with JWT's"
  (:require [environ.core :refer [env]]
            [clj-jwt.core  :refer :all]
            [clj-time.core :refer [now plus days]]))

(defn secret
  "Returns a secret based on environment variables. This secret should be shared
   and unique for all microservices of clanhr"
  ([]
   (secret env))
  ([env]
   (or
     (:clanhr-auth-secret env)
     (:secret env)
     (throw (Exception. "Can't resolve auth token")))))

(defn build-claim
  "Build a claim. the iss has the info that we want to share between the multiple
   parties, the exp is the expiration date whenever the token will no longer be valid,
   the iat is current timestamp"
  [user]
  {:iss user
   :exp (plus (now) (days 10))
   :iat (now)})

(defn token-for
  "Creates a JSON Web Token (JWT) for a user. This token has the following format:
                       aaaaaaaaaaa.bbbbbbbbb.ccccccccccccc
                          <header>.<payload>.<signature>
  Header: The header corries 2 parts, declaring the type, which is `JWT` and the hashing algorithm
          to use (HMAC SHA256 in this case)
          Here's an example: {'typ': 'JWT', 'alg': 'HS256'}

  Payload: The payload will carry the bulk of our JWT, also called the JWT Claims. This is where we
           will put the information that we want to transmit and other information about our token.
           Here's an example: {'id': '1234567890', 'name': 'John Doe', 'email': 'john.doe@mail.com'}

  Signature: The third and final part of our JWT is going to be the signature. This signature is made up
             of a hash of the following components: header, payload and secret.

  The JWT standard can be used across multiple languages and systems. You can use the token in a URL, POST parameter,
  or a HTTP header. The versatility of the JWT let's us authenticate an API easily by passing information through the
  token.

  The benefits of having a custom payload where you can store all the info that you want is pretty obvious, in a microservice
  architecture where you have a service to manage for instance expenses. This service can receive a request with the JWT inside of it
  and use the secret to decode and get all the info that the service needs. If the service can decode the token,
  means that the JWT was generated by our system (even in another service) with the same secret. which means that you
  can assume the user is properly loggedin. This means that the only thing that you need to share between our services is this
  `secret` thing to help on generate/decode our JWT's."

  ([args]
   (-> (build-claim (:user args))
       jwt
       (sign :HS256 (secret))
       to-str))
  ([user-email user-id account-id]
   (token-for {:user {:email user-email
                      :account account-id
                      :user-id user-id}})))

(defn system-token
  "Creates a tokens used by the system/services to communicate with each other"
  []
  (token-for {:user {:name "clanhr-system" :system true}}))

(defn payment-token
  "Creates a token used for payment flow"
  []
  (token-for {:user {:name "clanhr-payment" :payment true}}))

(defn parse
  "Parse token"
  [token]
  (-> token str->jwt))

(defn valid?
  "Check if token is valid"
  [token]
  (if token
    (verify token (secret))
    false))

(defn principal
  "Get the principal from the given token"
  [token]
  (if token
    (get-in token [:claims, :iss])
    false))
