(ns courier.redis-cache
  "Cache results in Redis using the optional dependency com.taoensso/carmine"
  (:require [clojure.string :as str]
            [courier.cache :as cache]
            [courier.fingerprint :as fingerprint]
            [courier.time :as time]))

(def ^:private carmine-available?
  (try
    (require 'taoensso.carmine)
    true
    (catch Throwable _ false)))

(defmacro wcar [& body]
  (when carmine-available?
    `(taoensso.carmine/wcar ~@body)))

(defn- redis [f & args]
  (apply (ns-resolve (symbol "taoensso.carmine") (symbol (name f))) args))

(defn- cache-key [spec params]
  (let [id (cache/cache-id spec)
        params (cache/get-cache-relevant-params spec params)]
    (->> [(namespace id)
          (name id)
          (fingerprint/fingerprint params)]
         (remove empty?)
         (str/join "/"))))

(defn create-redis-cache [conn-opts]
  (assert carmine-available? "com.taoensso/carmine needs to be on the classpath")
  (assert (not (nil? conn-opts)) "Please provide connection options")
  (reify cache/Cache
    (lookup [_ spec params]
      (wcar conn-opts (redis :get (cache-key spec params))))
    (put [_ spec params res]
      (let [ttl (- (time/millis (:expires-at res)) (time/millis (time/now)))
            cache-key (cache-key spec params)]
        (wcar conn-opts (redis :psetex cache-key ttl (assoc res ::cache/key cache-key)))
        {::cache/key cache-key}))))
