(ns courier.file-cache
  (:require [courier.cache :as cache]
            [courier.fingerprint :as f]
            [courier.fs :as fs]))

(defn str-id [spec]
  (let [id (cache/cache-id spec)]
    (str (namespace id) "." (name id))))

(defn filename [dir spec params]
  (let [fingerprinted-name (f/fingerprint (cache/get-cache-relevant-params spec params))
        [_ prefix postfix] (re-find #"(..)(.+)" fingerprinted-name)
        dirname (str dir "/" (str-id spec) "/" prefix)]
    (str dirname "/" postfix ".edn")))

(defn slurp-edn [file]
  (try
    (let [content (fs/read-file file)]
      (if-not (empty? content)
        #?(:clj (read-string content)
           :cljs (cljs.reader/read-string content))
        nil))
    (catch #?(:clj Throwable
              :cljs :default) e
      nil)))

(defn create-file-cache [{:keys [dir]}]
  (assert (string? dir) "Can't create file cache without directory")
  (fs/ensure-dir dir)
  (reify cache/Cache
    (lookup [_ spec params]
      (when-let [file (filename dir spec params)]
        (when-let [val (slurp-edn file)]
          (if (cache/expired? val)
            (do
              (fs/delete-file file)
              nil)
            val))))
    (put [_ spec params res]
      (when-let [file (filename dir spec params)]
        (fs/ensure-dir (fs/dirname file))
        (fs/write-file file (pr-str (assoc res ::cache/file-name file)))
        {::file-name file}))))
