(ns circleci.statsd
  (:require [clojure.string :as str]
            [clojure.tools.logging :refer (infof)]
            clj-statsd))

(defonce ^:private global-tags (atom {}))

(defn- reset-global-tags
  []
  (reset! global-tags {}))

(defn- maybe-deref
  "Call deref on anything that can be deref'ed. We call deref with a zero
   timeout since we don't want to block to wait for anything. If it's not ready
   by the time we attempt to deref it then we return nil."
  [x]
  (cond (instance? clojure.lang.IBlockingDeref x) (deref x 0 nil)
        (instance? clojure.lang.IDeref x) (deref x)
        :else x))

(defn- maybe-name
  "Attempt to (name x) and return x if that doesn't work."
  [x]
  (if (instance? clojure.lang.Named x)
    (name x)
    x))

(defn map->tags
  "Transform a map into a list of tag strings"
  [tag-map]
  (map (fn [[k v]]
         (if (nil? v)
           (str (maybe-name k))
           (str (maybe-name k) ":" (maybe-name v))))
       tag-map))

(defn- stringify-tags
  "Deref any promises/futures and call deref on anything that can be deref'ed so that "
  [tags]
  (->> tags
       (map maybe-deref)
       (filter some?)
       (map maybe-name)
       (map str)))

(defn- merge-global-tags
  "Given a clj-statsd style of tag list, merge in the global tags. If any of
   the given tags has the same name as a global tag then that global tag is
   omitted."
  [tags]
  (let [passed-tag-names (set (map #(str/replace % #":.*" "") tags))]
    (concat
     tags
     (->> @global-tags
          (remove (fn [[k v]]
                    (contains? passed-tag-names (maybe-name k))))
          map->tags))))

(defn- remove-invalid-tags
  "Remove tags which are not valid. For now, this is a very simple string?
   check but we can extend this to be more useful in the future."
  [tags]
  (filter string? tags))

(defn- process-tags
  "Process the tags we've been given so that they"
  [tags]
  (-> tags
      stringify-tags
      merge-global-tags
      remove-invalid-tags))

(defn setup
  [host port & {:as opts}]
  (when (:global-tags opts)
    (reset! global-tags (:global-tags opts)))
  (let [opts-list (apply concat (seq (dissoc opts :global-tags)))]
    (apply clj-statsd/setup host port opts-list)))

(defn increment
  ([k]   (clj-statsd/increment k 1 1.0 []))
  ([k v] (clj-statsd/increment k v 1.0 []))
  ([k v {:keys [rate tags]
         :or {rate 1.0, tags []}}]
    (clj-statsd/increment k v rate (process-tags tags))))

(defn decrement
  ([k]   (clj-statsd/decrement k 1 1.0 []))
  ([k v] (clj-statsd/decrement k v 1.0 []))
  ([k v {:keys [rate tags]
         :or {rate 1.0, tags []}}]
    (clj-statsd/decrement k v rate (process-tags tags))))

(defn timing
  ([k v] (clj-statsd/timing k v 1.0 []))
  ([k v {:keys [rate tags]
         :or {rate 1.0, tags []}}]
    (clj-statsd/timing k v rate (process-tags tags))))


(defn timer
  "Returns a delay which will yield the time in milliseconds between creation
   and first deref.

   Example usage:
   (let [t (timer)]
     (try
       ; do stuff
       (timing :stuff @t {:tags [:ok]})
       (catch Exception e
         (timing :stuff @t {:tags [:error]})
         (throw e))))"
  []
  (let [start-time (System/nanoTime)]
    (delay (-> (System/nanoTime)
             (- start-time)
             (quot 1000000)))))


(defn gauge
  ([k v] (clj-statsd/gauge k v 1.0 []))
  ([k v {:keys [rate tags]
         :or {rate 1.0, tags []}}]
    (clj-statsd/gauge k v rate (process-tags tags))))

(defn unique
  ([k v] (clj-statsd/unique k v []))
  ([k v {:keys [tags] :or {tags []}}]
    (clj-statsd/unique k v (process-tags tags))))

(defn- remove-custom-tags
  [tags]
  (remove (partial re-find #"^[Xx]-") tags))

(defn- publish-timing-metric
  [metric tags elapsed-time]
  (let [metric-tags (remove-custom-tags (process-tags tags))]
    (when (> (count tags) (count metric-tags))
      (infof "with-timing-metric %s %s: took %d milliseconds"
             metric
             tags
             elapsed-time))
    (timing metric elapsed-time {:tags metric-tags})))

(defn- time-since
  [previous-nanotime]
  (-> (System/nanoTime)
      (- previous-nanotime)
      (/ 1e6)
      (#(Math/round %))))

(defn with-timing-metric-call
  "Wrap a call to f with timing instrumentation.
  'metric' is a keyword naming the metrics.
  'tags' is a vector of strings to tag the metric with.

  Tags that are prefixed with X- (e.g. X-foo-bar) are removed before submitting
  the timing metric to statsd.
  The prefixed tags and timing data are logged, allowing for analysis via logs.

  Use this for tags that are not appropriate for statsd services such as
  Datadog, e.g. tags with no bounds on the number of values."
  [metric tags f]
  (let [start-time (System/nanoTime)]
    (try
     (f)
     (finally
      (publish-timing-metric metric tags (time-since start-time))))))

(defmacro with-timing-metric
  "Helper macro for calling with-timing-metric-call"
  [metric tags & body]
  `(with-timing-metric-call ~metric ~tags (fn [] ~@body)))

(defn locking-with-timing-call
  "Like with-timing-metric-call but to handle the case where you're using
  (locking lock (f)) and you want to time both the acquisition of the lock and
  the function called inside the lock body.

  The lock metric will be published with the same tags as the function, but
  with the inner metric name and the lock object itself added as tags.

  lock - The object to lock with.
  lock-metric - The name of the metric to record the lock wait time.
  metric - The metric for the timing of the code inside the lock.
  tags - The tags for both metrics
  f - The function to be run inside the lock"
  [lock lock-metric metric tags f]
  (let [start-time (System/nanoTime)]
    (locking lock
      (publish-timing-metric lock-metric
                             (conj tags (maybe-name metric) (str lock))
                             (time-since start-time))
      (with-timing-metric-call metric tags f))))

(defmacro locking-with-timing
  "Helper macro for calling locking-with-timing-call"
  [lock lock-metric metric tags & body]
  `(locking-with-timing-call ~lock ~lock-metric ~metric ~tags (fn [] ~@body)))
